describe("Turtle", function () {
    'use strict';
	var c,
        ref = document.createElement('canvas'),
        loadImage = function (file) {
            var a = new Image();
            a.src = file;
            a.addEventListener('load', function () {
                ref.getContext('2d').drawImage(a, 0, 0);
            });
        };
        
    beforeEach(function () {
		jasmine.DEFAULT_TIMEOUT_INTERVAL = 10000;
        jasmine.addMatchers(imagediff.jasmine);
        TurtleGraphics = { 
			defaults: {
				canvasID: 'turtlecanvas',
				animate: true,
				degrees: true
			},
			doneDelegates: [],
		};
        Sk.configure({
            output: function(s) {
                console.log(s);
            },
        });
        c = document.createElement('canvas');
        c.id = "turtlecanvas";
        c.width = "200";
        c.height = "200";
        c.style.display = "none";
        ref.id = "reffw100";
        ref.width = "200";
        ref.height = "200";
        document.body.appendChild(c);
    });
    
    afterEach(function () {
        document.body.removeChild(c);
    });
    
    it("should be able to draw the example on the home page", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle\nt = turtle.Turtle()\nc = [ \"red\", \"green\", \"blue\", \"yellow\" ]\nfor color in c:\n    t.color(color)\n    t.forward(50)\n    t.left(360/len(c))",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAMgAAADICAYAAACtWK6eAAAG90lEQVR4Xu3aPXJcZRBG4TtFYkggICCDEDJ7JbAMQlgB8g5MQA4rgCWYFQCRneGQgCogsojEvUbFj2AsulUz1Zf3UZXLhd3fjPqce/gk24fFBwIIHCVwwAYBBI4TEIinA4GXEBCIxwMBgXgGEOgRcIP0uDkVQkAgIaKt2SMgkB43p0IICCREtDV7BATS4+ZUCAGBhIi2Zo+AQHrcnAohIJAQ0dbsERBIj5tTIQQEEiLamj0CAulxcyqEgEBCRFuzR0AgPW5OhRAQSIhoa/YICKTHzakQAgIJEW3NHgGB9Lg5FUJAICGirdkjIJAeN6dCCAgkRLQ1ewQE0uPmVAgBgYSItmaPgEB63JwKISCQENHW7BEQSI+bUyEEBBIi2po9AgLpcXMqhIBAQkRbs0dAID1uToUQEEiIaGv2CAikx82pEAICCRFtzR4BgfS4ORVCQCAhoq3ZIyCQHjenQggIJES0NXsEBNLj5lQIAYGEiLZmj4BAetycCiEgkBDR1uwREEiPm1MhBAQSItqaPQIC6XFzKoSAQEJEW7NHQCA9bic59eTJu88vL+/de/Dgm5O8/llf9M2nz5cf33vtrO95gjcTyAmgdl/y6mq52s4eDi9+2vfHW98tyw/3d/987X6BfT9Ff//s/wxk2beXi99DXy52vsf2P6v/0wO2910EMs+gQAY5EcggGdefikAGORHIIBkCmSdDIPOcuEEGORHIIBlukHkyBDLPiRtkkBOBDJLhBpknQyDznLhBBjkRyCAZbpB5MgQyz4kbZJATgQyS4QaZJ0Mg85y4QQY5EcggGW6QeTIEMs+JG2SQE4EMkuEGmSdDIPOcuEEGORHIIBlukHkyBDLPiRtkkBOBDJLhBpknQyDznLhBBjkRyCAZbpB5MgQyz4kbZJATgQyS4QaZJ0Mg85y4QQY5EcggGW6QeTIEMs+JG2SQE4EMkuEGmSdDIPOcuEEGORHIIBlukHkyBDLPiRtkkBOBDJLhBpknQyDznLhBBjkRyCAZbpB5MgQyz4kbZJATgQyS4QaZJ0Mg85y4QQY5EcggGW6QeTIEMs+JG2SQE4EMkuEGmSdDIPOcuEEGORHIIBlukHkyBDLPiRvkTE7Wh//b9a1eX398fDgsX/3b2wrkTDIKbyOQAqy7jK4P/7P1/NvXr/F4/fnhGsr28x8fArkL4dOcFchpuP7jVdeH/2L9xU9u/Mbn16Fs8SwCOZOMwtsIpADrLqPrw//Oev77I6+xxfPp+uOn7ffXm2XfXi6Wqxd7Xux8j83FXaQ7WyOwRrJ97/H+kVM/r7/+hkBqTE89LZBTE/7L66+BfLD+55e3vuXOrRy2+9ANcqtmAzcI/KdALtdDr+4b3f0Pl+XXV5bLp5/tfRNfYp31SbzlS6xftq/a1+8/Hp31k/JmLyWw88t8P3Zv+Sb94brJozWO7fsQH4MICORMMo78Me8X17fGszN9Gt6mSEAgRWDd8Rt/Ufj1dRiPu6/n3HkICOQ8nLe/BNz+qcn2x7gfHfunJmf6VLxNgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcICKQAy2geAYHkObdxgYBACrCM5hEQSJ5zGxcI/AaC8zTYdteqRgAAAABJRU5ErkJggg==");
        TurtleGraphics.doneDelegates.push(function () {
            //1 for tolerance
            expect(cctx).toImageDiffEqual(refctx, 1);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });
    
    it("should be able to move 50 px forward", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle\nt = turtle.Turtle()\nt.forward(50)",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,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");
        TurtleGraphics.doneDelegates.push(function () {
            expect(cctx).toImageDiffEqual(refctx);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });
    
    it("should be able to change the color", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle\nt = turtle.Turtle()\nt.color('red')\nt.forward(50)",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,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");
        TurtleGraphics.doneDelegates.push(function () {
            expect(cctx).toImageDiffEqual(refctx);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });
    
    it("should be able to draw a circle", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle\nt = turtle.Turtle()\nt.circle(50, 360)",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAMgAAADICAYAAACtWK6eAAAOOUlEQVR4Xu1dSahsVxX9QYIOjFESOxxYGQiCYnTkIDYVQQ2CJJJBBAPWt8GOEAVBB4IVBVEiRkUlYsQKOIgoJqLYxIH1bUBxkAgOMrM+wQ4FGxRsIroW7x7fffVe/bq163a11zqwebde3X3O2Wvvdfp767JzTl0h8AJkfCvkOZAp5MktF7RCfg9DlpAL1XXLRTi7ywxBqwi8ELm9EXITZHJGziWQ/4zvGOAUXjPQz0rMg0JyMc+rIFdAXn7GzczrAci9Jkt7PjVB9seSAUxSzNZI8Xd8/h3kM5BlB0FLMlKmFXmurJnyV1x/AvIpCAnoFETABAkCVwXn7RUxSi4Xq1acLTlJ0WciUW6BvBlyea3gBa7Zq/Rdnz5t76wsE2R3aBmIX4VcXVNlADIQxxKErOO7ITfW6si63TGiOu6O/AAaJkhz0Ce49S4I5wJMj0E+C/kkZNU8m17vZJ1nFVnKEIxEJlHGWudeAdpWmAmyDaGjCTKHUvPq1r/g792Qj0IOZXxPG9ijUApRHsQ1h2SHYsN2T3VwhwlyaVCn+PpLELbETBxKkSiH2vrSjo9Dbq7soR3nIcsOYitFlibIZjeSCB+svv4F/s4gm5ZjDy0YXoMKfwRybVVxDhM57HJvsuZJE+R0aDN4vgh5RvUVA4dkyZjqjQDJ/zrIKqOhUZtMkJPITfHxO5AnQH4PuSFRr7EpRriXsqh6E/Yg1wvY3JgvJsgxVDNccr7B9HPIqyAqQw5O4kmSsizMeQk/yycT5CgEOAbnShUTd5+52qOY6jjwWMxUEYS6zSbIUUvJoyJMbjnPnXsvcLizwoPYEBPZpE6QQg7ubXADcCkbCScNn+FjGW5Kk0SZIHMEQVnG5eoNz085HSNgkgALVYJ8GrbfVsWCh1Wbm4U6SSRxUiTIFPHwA5OjcWdZJwmXgKWGoWoEmcDBD0G4rPl5yNsbh4n2jWV1i8veL4KsVOBQIwh3i3m84huQcipXxdf72smeg08y/hryfIjEHpESQUoreBHO5e6xhIP3ZUVNn73uI5CnQ74FeW2LeY82KxWCTOEBzju4nMvrLIcO+w4sHo+/rypUYj6iQhCOmZ8NyXzwsC+yzFEQl8eJKecjqXtiBYIUh/LIOodWTvsjUOZy6Ruc7AR5KWLhh0pDgv1jv1EObGi4Gsh0TdWbNFI8tJuyE6QMrb4Px/B0rlN7CCyQFc+w3QuZtZftuHLKTJApoObE/B+Q50K4euXUHgITZMWhFp9xTzthz0yQJRzHdfv04+T2Yn7nnObQ4ISdWJMk6VJWgpTeg8u6bOlSr7QMGJXcG3kU8kTIWyH3DFiXTorOShCezOXTce49OgmbE5l+F59eDfkJ5CXdF9dvCRkJwh7jVxWMT3Hv0XlApcY7I0H4uCzfgJh6daXzsN+tgNJjvwdqPNKTJmUkyAre4a65H4LqL0x58PN+CLHnvkialI0gZWOQS7rs+p36Q6A0TDx+kuasWzaC/BTOeTHEG4P9EaOUVE5Lp3orTDaC/AbeeibkZZAf9R8j0iVOYT03Ztl7sBdJkTIRhEMqrl5x74Pr8079I8D9Ju6spzmflYkgMziGr6rx04L9E6OUWFaz3oV/fG64arRXciaCpF1qbM/dnefE30x5H4S/zcih7sGnTARJ170fYHSVY/CcCz7rAOt/qspZCMI5x58q67LYdKjx9d9MfsgSTFM4hSsoFyC8dhoOgSWK5inqFEfgsxBkBodwgu7jJcMRo5S8wAUfpEpx7CQLQfijmm+DfAzy/uFjRLoGc1jPZ0RSnKTOQpDfwiH8ybSUzyQcGN1SDXezEOTHCKLrsox7D4wQ69U1QUbowFQTwxHiu0uVTJBd0OrpXu+B9AR0g2LKkvvfcO8VDe4f9S1Zhlip1t5HHTHNKpfGHyZIM4f7rt0QMEF2w6vzu9M4pHOk+ikgjT/cg/QTMGqlmCAj83iZpPstJuNwjAkyDj/8vxa/xNXzvA8yCq9MUAs+uJbivQBZhlh/hEOugrwe8pVRhIluJaYwPc3B0SwEeRBOeSXEr/oZnpgzVCHNk51ZCDKHU9IckBs+xveqQSpfZCFIabV83H2v2G5F+WvI5WaIj7u3Amc7maQa97YDyWC5lPmgCTKYC04XXFZOuNzLpV6n4RAoS7wpltyzDLEYDisI38mb6tWXw8V5qOTy0oY0P5iaiSALuDTNo56h8BxeKd2b9TMRpEzU/eK44YhSGqnzqAKvDz5lIkiZh/wLXnn8wXvmMA1IN8zNRBCGFMlxuechg7CrzD9SHDEpCGYjSMpX8A8S7rsXmhL7bAQprZiXe3cP8H010g2vCEg2gtCm4iify9o35Jvr34Jb74P8AfK05mrjvzMjQeaAneeyfOykv/grPwX9PRR5Q3/Fdl9SRoJMAFv5Gehrqh6leyR1S0iNd0aCMFQXEG4apvq9vJFysEzOU/bYWQlSn6yzF+Gk3al9BPgOLPbW/JvyiE9WgjAUlhC+hp9PGPJJQ6f2EZgjS873LkCm7Wc/fI6ZCfIWwPsFyGOQp7oXaT3YJsjxoar3uL5qkFovZOgMMxOE2JaXWqccHw8cPAuUz3le6rNv2QnCVq6saKVt5QYgCodTfDEDf3Kb873VAHXopcjsBCGIcwjHyal+4L6X6NhcCInxJMgdFb4DV6e74hUIwhUWkoMPU6V4DLS7cGiUc2lw/o27uWueeoVQgSD0+k2Q+yv3p1yObBTa+980RRYcWjFJDFlVCEKHlg2tFa5JktQt3/5cOJUDe2KuWk0g6YdWxXolgtBmDrWurVrBV3QQRJmzfADG3QhJ87x5E2epEYSt3yMQPnHopd8mEXJ0T+l9069arUOiRhDa/wHIhysgzuPvonmcSN45g9V8lSiTxLyj7mVFgtD+utNNks28fwO++rJyY6JKEPq8vKKG117ZOk0Srvx9HcIYkZmUe4h1EgEOr3hcgita3CPxcOsIn3oPK73BqtyDFKoUkvCzh1snySG/kGGCHNFkDuFxFKa7Ie9Y72pFPtdx8KkDON0EOY78+rDiZ/g3n61W2UzkJiBXqjjvcE9aaw1NkJNdA1vNOyGPg6wgfDMKx+CZE0/j8hjOBMJ9DpJkmdngXWwzQU6jxUDhrjF33JnmEK7iZEx8VQ9f2cPEHXKSY5XR0KhNJshm5MruMe9gL8LeZRkFemR6U9TnLgh7DyYOr940sjqOojomyKXdwEBaQHhUnonXJMqhzk041yAxZpU9j+LvhyD3jCIaR1gJE2S7UxhU3FQsq1x8Qfa3IVwSPhSi0IbbKzt4zSS7+bfd5cd3mCDN0ZrgVg5FppUKybGA8N1bqwbZcJh2JYQ9EOc4fSTWmcSYQQoxLlSfm9S5jzqOugwTZHf38G0pDLjraqokCoOeLzDYlBiQZai2xDVbcP7tInGyfRukfqSfxJh3WGYXdgyepwkSd8GkCjgeVSmJJCBRuAO9vjzM4CzDtHL/oiIK9fZNnHCzLjNI6S2YJ58A5AS8jTL2rePB6Zsg+7uMROEcha126SGY638g34QsIWy9OSQrb1hZL3WOf3CotsuchoTgi/GmldRJcRH/4yocyWpi7OFjE2QP8M5QZdDOIO+E8Jeu6onBz5fYXb2hSH5fiMJbJpBCOAY/8+b/KCTFeiIpSIgFZL332lCk/70NARNkG0Lx70sgM5gp9d4lnuux5j9xyY2+ZSWrNjJ1HicRMEH6iwgS5lZIeZpxW8nsEepBTyLwM4U9xC7DsW1l+fsNCJgg/YYGh0B88cFZieeg5hDOHZxGgoAJ0p8j2INsmqRzyZfEcK/Qnz8alWSCNIKplZvYO6wv83I5mP9ftVKCM2kdAROkdUg3ZkgSlIm6N+36w32vkkyQveDbSZkTay7Xcs+kr6MmO1XQN59GwARxVBiBSyBggjg8jIAJ4hgwAjEE3IPEcLOWCAImiIijbWYMARMkhpu1RBAwQUQcbTNjCJggMdysJYKACSLiaJsZQ8AEieFmLREETBARR9vMGAImSAw3a4kgYIKIONpmxhAwQWK4WUsEARNExNE2M4aACRLDzVoiCJggIo62mTEETJAYbtYSQcAEEXG0zYwhYILEcLOWCAImiIijbWYMARMkhpu1RBAwQUQcbTNjCJggMdysJYKACSLiaJsZQ8AEieFmLREETBARR9vMGAImSAw3a4kgYIKIONpmxhAwQWK4WUsEARNExNE2M4aACRLDzVoiCJggIo62mTEETJAYbtYSQcAEEXG0zYwhYILEcLOWCAImiIijbWYMARMkhpu1RBAwQUQcbTNjCJggMdysJYKACSLiaJsZQ8AEieFmLREETBARR9vMGAImSAw3a4kgYIKIONpmxhAwQWK4WUsEARNExNE2M4aACRLDzVoiCJggIo62mTEETJAYbtYSQcAEEXG0zYwhYILEcLOWCAImiIijbWYMARMkhpu1RBAwQUQcbTNjCJggMdysJYKACSLiaJsZQ8AEieFmLREETBARR9vMGAImSAw3a4kgYIKIONpmxhAwQWK4WUsEARNExNE2M4aACRLDzVoiCJggIo62mTEETJAYbtYSQcAEEXG0zYwhYILEcLOWCAImiIijbWYMARMkhpu1RBAwQUQcbTNjCJggMdysJYKACSLiaJsZQ8AEieFmLREETBARR9vMGAImSAw3a4kgYIKIONpmxhAwQWK4WUsEARNExNE2M4aACRLDzVoiCJggIo62mTEETJAYbtYSQcAEEXG0zYwhYILEcLOWCAImiIijbWYMARMkhpu1RBAwQUQcbTNjCJggMdysJYKACSLiaJsZQ8AEieFmLREETBARR9vMGAImSAw3a4kgYIKIONpmxhAwQWK4WUsEgf8BirDw2P5wajkAAAAASUVORK5CYII=");
		TurtleGraphics.doneDelegates.push(function () {
            expect(cctx).toImageDiffEqual(refctx);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });
    
	it("it should be able to stamp a grid", function (done) {
		var cctx = c.getContext("2d"),
			code = "import turtle\n\nt = turtle.Turtle()\nt.up()\nt.speed(10)\nt.left(30)\nfor i in range(-90, 110, 20):\n    for j in range(-90, 110, 20):\n        t.goto(i, j)\n        t.color(i,j,i+j)\n        t.stamp()",
			refctx = ref.getContext("2d");
		loadImage("data:image/png;base64,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");
		TurtleGraphics.doneDelegates.push(function () {
			expect(cctx).toImageDiffEqual(refctx, 16);
			done();
		});
		Sk.importMainWithBody("", false, code);
	});
    
	it("it should be able to run \"Our First Turtle Program\"", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle               # allows us to use the turtles library\nwn = turtle.Screen()        # creates a graphics window\nalex = turtle.Turtle()      # create a turtle named alex\nalex.forward(150/2)         # tell alex to move forward by 150 units\nalex.left(90)               # turn by 90 degrees\nalex.forward(75/2)          # complete the second side of a rectangle",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,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");
        TurtleGraphics.doneDelegates.push(function () {
            expect(cctx).toImageDiffEqual(refctx);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });
    
	it("it should be able to run \"The second example\"", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle\n\nwn = turtle.Screen()\nwn.bgcolor(\"lightgreen\")        # set the window background color\n\ntess = turtle.Turtle()\ntess.color(\"blue\")              # make tess blue\ntess.pensize(3)                 # set the width of her pen\n\ntess.forward(50)\ntess.left(120)\ntess.forward(50)\n\nwn.exitonclick()                # wait for a user click on the canvas",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,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");
        TurtleGraphics.doneDelegates.push(function () {
            expect(cctx).toImageDiffEqual(refctx);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });

	it("it should be able to draw \"A herd of turtles\"", function (done) {
		var cctx = c.getContext("2d"),
			code = "import turtle\nwn = turtle.Screen()             # Set up the window and its attributes\nwn.bgcolor(\"lightgreen\")\n\n\ntess = turtle.Turtle()           # create tess and set some attributes\ntess.color(\"hotpink\")\ntess.pensize(5)\n\nalex = turtle.Turtle()           # create alex\n\ntess.forward(80)                 # Let tess draw an equilateral triangle\ntess.left(120)\ntess.forward(80)\ntess.left(120)\ntess.forward(80)\ntess.left(120)                   # complete the triangle\n\ntess.right(180)                  # turn tess around\ntess.forward(80)                 # move her away from the origin\n\nalex.forward(50)                 # make alex draw a square\nalex.left(90)\nalex.forward(50)\nalex.left(90)\nalex.forward(50)\nalex.left(90)\nalex.forward(50)\nalex.left(90)\n",
			refctx = ref.getContext("2d");
		loadImage("data:image/png;base64,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");
		TurtleGraphics.doneDelegates.push(function () {
			expect(cctx).toImageDiffEqual(refctx, 16);
			done();
		});
		Sk.importMainWithBody("", false, code);
	});
    
    it("it should be able to run \"Iteration Simplifies our Turtle Program\"", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle            #set up alex\nwn = turtle.Screen()\nalex = turtle.Turtle()\n\nfor i in [0,1,2,3]:      #repeat four times\n    alex.forward(50)\n    alex.left(90)\n\nwn.exitonclick()",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,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");
        TurtleGraphics.doneDelegates.push(function () {
            expect(cctx).toImageDiffEqual(refctx);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });
    
    it("it should be able to create a spiral of turtles.", function (done) {
        var cctx = c.getContext("2d"),
            code = "import turtle\ntess = turtle.Turtle()\ntess.color(\"blue\")\ntess.shape(\"turtle\")\n\ntess.up()                       # this is new\nfor size in range(5,60,2):      # start with size = 5 and grow by 2\n    tess.stamp()                # leave an impression on the canvas\n    tess.forward(size)          # move tess along\n    tess.right(24)              # and turn her",
            refctx = ref.getContext("2d");
        loadImage("data:image/png;base64,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");
        TurtleGraphics.doneDelegates.push(function () {
            expect(cctx).toImageDiffEqual(refctx);
            done();
        });
		Sk.importMainWithBody("", false, code);
    });
	
	it("should be able to run the example from \"How to Think Like a Computer Scientist\"", function (done) {
		var cctx = c.getContext("2d"),
			code = "import turtle\nimport random\n\ndef main():\n    tList = []\n    head = 0\n    numTurtles = 10\n    for i in range(numTurtles):\n        nt = turtle.Turtle()   # Make a new turtle, initialize values\n        nt.setheading(head)\n        nt.pensize(2)\n        nt.speed(10)\n        nt.tracer(30,0)\n        tList.append(nt)       # Add the new turtle to the list\n        head = head + 360/numTurtles\n\n    for i in range(100):\n        moveTurtles(tList,15,i)\n\n    w = tList[0]\n    w.up()\n    w.goto(-130/2,40/2)\n    w.write(\"How to Think Like a \",True,\"center\",\"30px Arial\")\n    w.goto(-130/2,-35/2)\n    w.write(\"Computer Scientist\",True,\"center\",\"30px Arial\")\n\ndef moveTurtles(turtleList,dist,angle):\n    for turtle in turtleList:   # Make every turtle on the list do the same actions.\n        turtle.forward(dist/2)\n        turtle.right(angle/2)\n\nmain()",
			refctx = ref.getContext("2d");
		loadImage("data:image/png;base64,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");
		TurtleGraphics.doneDelegates.push(function () {
			expect(cctx).toImageDiffEqual(refctx);
			done();
		});
		Sk.importMainWithBody("", false, code);
	});
});

describe("Turtle (non-animated)", function () {
	'use strict';
	var c,
		ref = document.createElement('canvas'),
		loadImage = function (file, done) {
			var a = new Image();
			a.src = file;
			a.addEventListener('load', function () {
				ref.getContext('2d').drawImage(a, 0, 0);
				if (done) { done(); }
			});
		};

	beforeEach(function () {
		jasmine.DEFAULT_TIMEOUT_INTERVAL = 10000;
		jasmine.addMatchers(imagediff.jasmine);
		TurtleGraphics = { 
			doneDelegates: [],
			defaults: {
				animate: false,
				canvasID: 'turtlecanvas',
				degrees: true,
			}
		};
		Sk.configure({
			output: function(s) {
				console.log(s);
			},
		});
		c = document.createElement('canvas')
		c.id = "turtlecanvas";
		c.width = "200";
		c.height = "200";
		c.style.display = "none";
		ref.id = "reffw100";
		ref.width = "200";
		ref.height = "200";
		document.body.appendChild(c);
	});

	afterEach(function () {
		document.body.removeChild(c);
	});

	it("should be able to draw the example on the home page", function (done) {
		var cctx = c.getContext("2d"),
			code = "import turtle\nt = turtle.Turtle()\nc = [ \"red\", \"green\", \"blue\", \"yellow\" ]\nfor color in c:\n    t.color(color)\n    t.forward(50)\n    t.left(360/len(c))",
			refctx = ref.getContext("2d");
		loadImage("data:image/png;base64,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", function() {
			Sk.importMainWithBody("", false, code);
			expect(cctx).toImageDiffEqual(refctx, 1);
			done();
		});
	});
	
	it("should be able to go foward 50 px", function (done) {
		var cctx = c.getContext("2d"),
			code = "import turtle\nt = turtle.Turtle()\nt.forward(50)",
			refctx = ref.getContext("2d");
		loadImage("data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAMgAAADICAYAAACtWK6eAAAFiElEQVR4Xu3V21EcUBDEUIgIyD8Im4zAAcBH8yiV5/A/u32lVfH44A8BBD4k8IgNAgh8TEAgfh0IfEJAIH4eCAjEbwCBjYD/IBs3V0cICOSIaM/cCAhk4+bqCAGBHBHtmRsBgWzcXB0hIJAjoj1zIyCQjZurIwQEckS0Z24EBLJxc3WEgECOiPbMjYBANm6ujhAQyBHRnrkREMjGzdURAgI5ItozNwIC2bi5OkJAIEdEe+ZGQCAbN1dHCAjkiGjP3AgIZOPm6ggBgRwR7ZkbAYFs3FwdISCQI6I9cyMgkI2bqyMEBHJEtGduBASycXN1hIBAjoj2zI2AQDZuro4QEMgR0Z65ERDIxs3VEQICOSLaMzcCAtm4uTpCQCBHRHvmRkAgGzdXRwgI5Ihoz9wICGTj5uoIAYEcEe2ZGwGBbNxcHSEgkCOiPXMjIJCNm6sjBARyRLRnbgQEsnFzdYSAQI6I9syNgEA2bq6OEBDIEdGeuREQyMbN1RECAjki2jM3AgLZuLk6QkAgR0R75kZAIBs3V0cICOSIaM/cCAhk4+bqCAGBHBHtmRsBgWzcXB0hIJAjoj1zIyCQjZurIwQEckS0Z24EBLJxc3WEgECOiPbMjYBANm6ujhAQyBHRnrkREMjGzdURAgI5ItozNwIC2bi5OkJAIEdEe+ZGQCAbN1dHCAjkiGjP3AgIZOPm6ggBgRwR7ZkbAYFs3FwdISCQI6I9cyMgkI2bqyMEBHJEtGduBASycXN1hIBAjoj2zI2AQDZuro4QEMgR0Z65ERDIxs3VEQICOSLaMzcCAtm4uTpCQCBHRHvmRkAgGzdXRwgI5Ihoz9wICGTj5uoIAYEcEe2ZGwGBbNxcHSEgkCOiPXMjIJCNm6sjBARyRLRnbgQEsnFzdYSAQI6I9syNgEA2bq6OEBDIEdGeuREQyMbN1RECAjki2jM3AgLZuP3U1Z/3D37+qQ//5c/9+/59L7/8nd/+dQL5dqRf+sB/P6qnL31C5/j1f4hdIJ0flCVBAgIJSjGpQ0AgHReWBAkIJCjFpA4BgXRcWBIkIJCgFJM6BATScWFJkIBAglJM6hAQSMeFJUECAglKMalDQCAdF5YECQgkKMWkDgGBdFxYEiQgkKAUkzoEBNJxYUmQgECCUkzqEBBIx4UlQQICCUoxqUNAIB0XlgQJCCQoxaQOAYF0XFgSJCCQoBSTOgQE0nFhSZCAQIJSTOoQEEjHhSVBAgIJSjGpQ0AgHReWBAkIJCjFpA4BgXRcWBIkIJCgFJM6BATScWFJkIBAglJM6hAQSMeFJUECAglKMalDQCAdF5YECQgkKMWkDgGBdFxYEiQgkKAUkzoEBNJxYUmQgECCUkzqEBBIx4UlQQICCUoxqUNAIB0XlgQJCCQoxaQOAYF0XFgSJCCQoBSTOgQE0nFhSZCAQIJSTOoQEEjHhSVBAgIJSjGpQ0AgHReWBAkIJCjFpA4BgXRcWBIkIJCgFJM6BATScWFJkIBAglJM6hAQSMeFJUECAglKMalDQCAdF5YECQgkKMWkDgGBdFxYEiQgkKAUkzoEBNJxYUmQgECCUkzqEBBIx4UlQQICCUoxqUNAIB0XlgQJCCQoxaQOAYF0XFgSJCCQoBSTOgQE0nFhSZCAQIJSTOoQEEjHhSVBAgIJSjGpQ0AgHReWBAkIJCjFpA4BgXRcWBIkIJCgFJM6BATScWFJkIBAglJM6hAQSMeFJUECAglKMalDQCAdF5YECQgkKMWkDgGBdFxYEiQgkKAUkzoEBNJxYUmQgECCUkzqEBBIx4UlQQICCUoxqUNAIB0XlgQJCCQoxaQOAYF0XFgSJCCQoBSTOgQE0nFhSZCAQIJSTOoQEEjHhSVBAgIJSjGpQ0AgHReWBAm8AWn6BMka20FaAAAAAElFTkSuQmCC", function () {
			Sk.importMainWithBody("", false, code);
			expect(cctx).toImageDiffEqual(refctx);
			done();
		});
	});
});