implementation module StdInt

// ****************************************************************************************
//	Concurrent Clean Standard Library Module Version 1.3
//	Copyright 1998 University of Nijmegen
// ****************************************************************************************

import StdOverloaded
from StdMisc import abort
from StdBool import &&,otherwise
from StdChar import <
from StdClass import <=,not
import StdArray

instance + Int
where
	(+) :: !Int !Int -> Int
	(+) a b
		= code {
		.inline +;i
			addI
		.end
		}

instance - Int
where
	(-) :: !Int !Int -> Int
	(-) a b
		= code {
		.inline -;i
			subI
		.end
		}

instance zero Int
where
 zero:: Int
 zero
	= code {
		.inline zero;i
			pushI 0
		.end
	}

instance * Int
where
 (*) :: !Int !Int -> Int
 (*) a b
	= code {
		.inline *;i
			mulI
		.end
	}

instance / Int
where
 (/) :: !Int !Int -> Int
 (/) a b
	= code {
		.inline /;i
			divI
		.end
	}

instance one Int
where
 one :: Int
 one
	= code {
		.inline one;i
			pushI 1
		.end
	}

instance ^ Int
where
 (^) :: !Int !Int -> Int
 (^) x n
	|	n < 0
		= abort "^ (Int) called with negative power argument"
	// otherwise
		= pow x n
	where
		pow :: !Int !Int -> Int
		pow _ 0
			=	1
		pow a b
			| isEven b
				=	x * x
			// otherwise
				=	a * x * x
			where
				x
					= pow a (b >> 1)

instance abs Int
where
 abs::!Int -> Int
 abs x = if (x<0) (0 - x) x

instance sign Int
where
 sign::!Int -> Int
 sign x | x == 0	= 0
		| x < 0		= -1
					= 1

instance ~ Int
where
 ~ :: !Int -> Int
 ~ x
	= code {
		.inline ~;i
			pushI 0
			subI
		.end
	}
	
instance ==	Int
where
 (==) :: !Int !Int -> Bool
 (==) a b
	= code {
		.inline ==;i
			eqI
		.end
	}

instance < Int
where
 (<) :: !Int !Int -> Bool
 (<) a b
	= code {
		.inline <;i
			ltI
		.end
	}

instance toInt Int
where
 toInt :: !Int -> Int //	dummy
 toInt a
	= code {
		.inline toInt;i
			no_op
		.end
	}

instance toInt Char
where
 toInt::!Char -> Int
 toInt a
	= code {
		.inline toInt;c
			CtoI
		.end
	}

instance toInt Real
where
 toInt :: !Real -> Int
 toInt a
	= code {
		.inline toInt;r
			 RtoI
		.end
	}

instance fromInt Int
where
 fromInt :: !Int -> Int
 fromInt a
	= code {
		.inline fromInt;i
			no_op
		.end
	}

instance fromInt Char
where
 fromInt :: !Int -> Char
 fromInt a
	= code {
		.inline fromInt;c
			ItoC
		.end
	}

instance fromInt Real
where
 fromInt :: !Int -> Real
 fromInt a
	= code {
		.inline fromInt;r
			ItoR
		.end
	}

instance fromInt {#Char}
where
 fromInt :: !Int -> {#Char}
 fromInt a
	= code {
		.inline fromInt;#
		.d 0 1 i
			jsr ItoAC
		.o 1 0
		.end
	}

// Additional functions for integer arithmetic: 

instance mod Int
where
	(mod) a b
	= code {
		.inline mod;i
			modI
		.end
	  }

instance rem Int
where
	(rem) a b
	= code {
		.inline rem;i
			push_b 1
			push_b 2
			push_b 2
			divI
			mulI
			update_b 0 2
			pop_b 1
			subI
		.end
	  }

instance gcd Int where
	gcd x y    = gcdnat (abs x) (abs y)
	where
		gcdnat x 0 = x
	    gcdnat x y = gcdnat y (x mod y)

instance lcm Int
where
	lcm _ 0    = 0
	lcm 0 _    = 0
	lcm x y    = abs ((x / gcd x y) * y)

//	Test on Integers:

instance isEven Int
where
	isEven a
		= code {
		.inline isEven;i
			pushI 1
			and%
			pushI 0
			eqI
		.end
		}

instance isOdd Int
where
	isOdd a
		= code {
		.inline isOdd;i
			pushI 1
			and%
			pushI 0
			eqI
			notB
		.end
		}

//	Operators on Bits:

(bitor) infixl 6 :: !Int !Int -> Int
(bitor) a b
	= code {
		.inline bitor
			or%
		.end
	}

(bitand) infixl 6 ::	!Int !Int -> Int
(bitand) a b
	= code {
		.inline bitand
			and%
		.end
	}

(bitxor) infixl 6 :: !Int !Int -> Int
(bitxor) a b
	= code {
		.inline bitxor
			xor%
		.end
	}

(<<) infix 7 :: !Int !Int -> Int
(<<) a b
	= code {
		.inline <<
			shiftl%
		.end
	}

(>>) infix 7 :: !Int !Int -> Int
(>>) a b
	= code {
		.inline >>
			shiftr%
		.end
	}

instance toInt {#Char}
where
 toInt::!{#Char} -> Int
 toInt s
	| len==0
		=	0
	| neg
		=	~signedval
	| pos
		=	signedval
	// otherwise
		=	other
 where
	len	=	size s
	neg	=	s.[0] == '-'
	pos	=	s.[0] == '+'

	signedval
			=	toInt2 1 0 s
	other
			=	toInt2 0 0 s

	toInt2:: !Int !Int !{#Char} -> Int
	toInt2 posn val s
		|	len==posn		=	val
		|	isdigit			=	toInt2 (posn+1) (n+val*10) s
		//	otherwise
							=	0
	where
		n 		=	toInt (s.[posn]) - toInt '0'
		isdigit	=	0<=n && n<= 9

bitnot :: !Int -> Int
bitnot a
	= code {
		.inline bitnot
			not%
		.end

		.inline lcm;i
		.end
		.inline gcd;i
		.end
		.inline abs;i
		.end
		.inline sign;i
		.end
		.inline ^;i
		.end
		.inline toInt;#
		.end
	}
