definition module SQL

// *********************************************************************************************************************
// Clean Relational Database (SQL) API. v 0.1
//
// This module defines the types and functions for working with
// relational databases. 
// *********************************************************************************************************************

from StdTime import :: Date{..}, :: Time{..}
import StdString, StdMaybe

// *********************************************************************************************************************
// Basic types
// *********************************************************************************************************************

// SQL Statements and queries are just strings
:: SQLStatement		:== String

// This type represents the possible values that elements of a row in a 
// result set can have. The constructors map directly to the constructors of
// the SQLType type.
:: SQLValue			=	SQLVChar		String
					|	SQLVVarchar 	String
					|	SQLVText		String
					|	SQLVInteger		Int
					|	SQLVReal		Real
					|	SQLVFloat		Real
					|	SQLVDouble		Real
					|	SQLVDate		Date
					|	SQLVTime		Time
					|	SQLVTimestamp	Int
					|	SQLVDatetime	Date Time
					|	SQLVEnum		String
					|	SQLVNull
					|	SQLVUnknown		String

// A row is just a list of values
:: SQLRow			:== [SQLValue]

// *********************************************************************************************************************
// Errors
// *********************************************************************************************************************

// Warning:				Non fatal errors, you can still continue
// InterfaceError:		Error related to the interface, not the database itself
// DatabaseError:		Error related to the database that can not be classified
//						as Operational error or Internal error
// DataError:			Error due to problems with the data
// OperationalError:	Error due to operational problems with the database.
//						E.g. disconnects, memory full etc.
// IntegrityError:		Errors related to data integrity, e.g. key constraint 
//						violations
// InternalError:		Errors related to internal problems in the database 
//						library
// ProgrammingError:	Errors of the end user, e.g. syntax errors in SQL
//						statements
// NotSupportedError:	An operation is not supported by the database library

:: SQLError	= SQLWarning			Int String	
			| SQLInterfaceError		Int String	
			| SQLDatabaseError		Int String	
			| SQLDataError			Int String	
			| SQLOperationalError	Int String
			| SQLIntegrityError		Int String
			| SQLInternalError		Int	String
			| SQLProgrammingError 	Int String
			| SQLNotSupportedError


// *********************************************************************************************************************
// Database Interaction API
// *********************************************************************************************************************

class SQLEnvironment env ctx
where
	sql_init			:: !*env									-> (!(Maybe SQLError), !(Maybe *ctx), !*env)
	sql_end				:: !*ctx !*env								-> (!(Maybe SQLError), !*env)					

class SQLContext ctx con
where
	sql_openConnection	:: !String !String !String !String !*ctx	-> (!(Maybe SQLError), !(Maybe *con), !*ctx)	
	sql_closeConnection	:: !*con !*ctx								-> (!(Maybe SQLError), !*ctx)
	
class SQLConnection con cur
where
	sql_openCursor		:: !*con									-> (!(Maybe SQLError), !(Maybe *cur), !*con)
	sql_closeCursor		:: !*cur !*con								-> (!(Maybe SQLError), !*con)

class SQLCursor cur
where
	sql_execute			:: !SQLStatement ![SQLValue] !*cur			-> (!(Maybe SQLError), !*cur)
	sql_executeMany		:: !SQLStatement ![[SQLValue]] !*cur		-> (!(Maybe SQLError), !*cur)
	sql_numRows			:: !*cur									-> (!(Maybe SQLError), !Int, !*cur)
	sql_numFields		:: !*cur									-> (!(Maybe SQLError), !Int, !*cur)
	sql_insertId		:: !*cur									-> (!(Maybe SQLError), !Int, !*cur)
	sql_fetchOne		:: !*cur									-> (!(Maybe SQLError), !(Maybe SQLRow), !*cur)
	sql_fetchMany		:: !Int !*cur 								-> (!(Maybe SQLError), ![SQLRow], !*cur)
	sql_fetchAll		:: !*cur									-> (!(Maybe SQLError), ![SQLRow], !*cur)
	sql_commit			:: !*cur									-> (!(Maybe SQLError), !*cur)
	sql_rollback		:: !*cur									-> (!(Maybe SQLError), !*cur)

// *********************************************************************************************************************
// Table definitions
// *********************************************************************************************************************

:: SQLDatabaseName	:== String
:: SQLTableName		:== String
:: SQLColumnName	:== String

// This type represents the possible data types that columns in an SQL table, 
// view or other list of columns can have.
:: SQLType			=	SQLTChar Int
					|	SQLTVarchar Int
					|	SQLTText
					|	SQLTInteger
					|	SQLTReal
					|	SQLTFloat
					|	SQLTDouble
					|	SQLTDate
					|	SQLTTime
					|	SQLTTimestamp
					|	SQLTDatetime
					|	SQLTEnum [String]
					|	SQLTUnknown Int

// A table consists of a name, a non-empty list of columns, a primary key and a 
// possibly empty list of foreign keys.
:: SQLTable			= {	name			:: SQLTableName
					,	columns			:: [SQLColumn]
					,	primary_key		:: SQLPrimaryKey
					,	foreign_keys	:: [SQLForeignKey]
					}
// A column consists of a name, a data type, an boolean which states whether
// NULL values are alowed and optionally a default value.
:: SQLColumn		= {	name		::	SQLColumnName
					,	type		::	SQLType
					,	null		::	Bool
					,	default		::	Maybe SQLValue
					}

// A primary key is just a list of column names which make up a primary key
:: SQLPrimaryKey	:== [SQLColumnName]

// A foreign key is a list of column names, a table which is referenced, a list
// of columns which is referenced and the action which should be taken when the
// key constraint is violated.
:: SQLForeignKey	= {	columns			::	[SQLColumnName]
					,	ref_table		::	SQLTableName
					,	ref_columns		::	[SQLColumnName]
					,	update_action	::	SQLRefAction
					,	delete_action	::	SQLRefAction
					}

:: SQLRefAction		= SQLCascade
					| SQLRestrict 
					| SQLNoAction
					| SQLSetNull
					| SQLSetDefault

class SQLTableInfo con
where
	sql_tables			:: !*con					-> (!(Maybe SQLError), ![SQLTableName], !*con)
	sql_describe		:: !SQLTableName !*con		-> (!(Maybe SQLError), !(Maybe SQLTable), !*con)

class SQLTableManagement cur
where
	sql_mkCreateTable	:: !SQLTable !*cur			-> (!SQLStatement, !*cur)
	sql_mkDropTable		:: !SQLTable !*cur			-> (!SQLStatement, !*cur)

	sql_createTable		:: !SQLTable !*cur			-> (!(Maybe SQLError), !*cur)
	sql_dropTable		:: !SQLTable !*cur			-> (!(Maybe SQLError), !*cur)

class SQLDatabaseManagement con
where
	sql_createDatabase	:: !SQLDatabaseName !*con	-> (!(Maybe SQLError), !*con)
	sql_dropDatabase	:: !SQLDatabaseName	!*con	-> (!(Maybe SQLError), !*con)

// *********************************************************************************************************************
// Utitility functions
// *********************************************************************************************************************

instance toString SQLType
instance toString SQLValue
instance toString SQLError

instance == SQLType
instance == SQLValue
