implementation module MySQL
//MySQL implementation of the Clean SQL database API
// 
import SQL
import StdEnv, StdMaybe
import code from "_MySQL.o"

//Abstract state types
:: MySQLContext		:== Int //This is not used to store info, just needed for Clean
:: MySQLConnection	:==	Int //This is the MYSQL* connection struct of MySQL
:: MySQLCursor		:== Int //This is our own cursor emulation, because MySQL does not use cursors

//Constants for the types defined in SQL.dcl
SQLTYPE_CHAR		:==  1
SQLTYPE_VARCHAR		:==	 2
SQLTYPE_TEXT		:==	 3
SQLTYPE_INTEGER		:==	 4
SQLTYPE_REAL		:==  5
SQLTYPE_FLOAT		:==  6
SQLTYPE_DOUBLE		:==  7
SQLTYPE_DATE		:==  8
SQLTYPE_TIME		:==  9
SQLTYPE_TIMESTAMP	:== 10
SQLTYPE_DATETIME	:== 11
SQLTYPE_ENUM		:== 12
SQLTYPE_NULL		:==  0
SQLTYPE_UNKNOWN		:== -1

//Error fetching
sql_connectionErrorC :: !*MySQLConnection -> (!Int, !Int, !*MySQLConnection)
sql_connectionErrorC a0 = code {
	ccall sql_connectionErrorC "I:VIII"
}
sql_cursorErrorC :: !*MySQLCursor -> (!Int, !Int, !*MySQLCursor)
sql_cursorErrorC a0 = code {
	ccall sql_cursorErrorC "I:VIII"
}
sql_connectionErrorMessageC :: !String !*MySQLConnection -> *MySQLConnection
sql_connectionErrorMessageC a0 a1 = code {
	ccall sql_connectionErrorMessageC "SI:I"
}
sql_cursorErrorMessageC :: !String !*MySQLCursor -> *MySQLCursor
sql_cursorErrorMessageC a0 a1 = code {
	ccall sql_cursorErrorMessageC "SI:I"
}

instance SQLEnvironment World MySQLContext
where
	sql_init :: !*World -> (!(Maybe SQLError),!(Maybe *MySQLContext),!*World)
	sql_init world
		# (context, world) = sql_initC world
		| context == 0	= (Nothing, Just context, world)
						= (Just (SQLInterfaceError 1 "Could not initialize the MySQL library"), Nothing, world) 
	where
		sql_initC :: !*World -> (!*MySQLContext,!*World);
		sql_initC a0 = code {
			ccall sql_initC ":I:A"
		}

	sql_end :: !*MySQLContext !*World -> (!(Maybe SQLError), !*World)
	sql_end context world = (Nothing, sql_endC context world)
	where
		sql_endC :: !*MySQLContext !*World -> *World;
		sql_endC a0 a1 = code {
			ccall sql_endC "I:V:A"
			fill_a 0 1
			pop_a 1
		}

instance SQLContext MySQLContext MySQLConnection
where
	sql_openConnection	:: !String !String !String !String !*MySQLContext	-> (!(Maybe SQLError),!(Maybe *MySQLConnection),!*MySQLContext)
	sql_openConnection host user password database context
		# (conn, context) = sql_openConnectionC host user password database context
		| conn == 0		= (Just (SQLInterfaceError 1 "Could not initialize a connection"), Nothing, context)
		| otherwise		
			# (errno,errsize,conn)	= sql_connectionErrorC conn
			| errno == 0			= (Nothing, Just conn, context)
			| otherwise
				# errmsg			= createArray errsize '\0'
				# conn				= sql_connectionErrorMessageC errmsg conn		//Dirty side effect
				# (_,context)		= sql_closeConnection conn context				//Free the connection
				= (Just (SQLDatabaseError errno errmsg), Nothing, context)	
	where
		sql_openConnectionC :: !String !String !String !String !*MySQLContext -> (!*MySQLConnection,!*MySQLContext);
		sql_openConnectionC a0 a1 a2 a3 a4 = code {
			ccall sql_openConnectionC "SSSS:I:I"
		}

	sql_closeConnection	:: !*MySQLConnection !*MySQLContext -> (!(Maybe SQLError), !*MySQLContext)
	sql_closeConnection connection context = (Nothing, sql_closeConnectionC connection context)
	where
		sql_closeConnectionC :: !*MySQLConnection !*MySQLContext -> *MySQLContext;
		sql_closeConnectionC a0 a1 = code {
			ccall sql_closeConnectionC "I:V:I"
		}


instance SQLConnection MySQLConnection MySQLCursor
where
	sql_openCursor :: !*MySQLConnection -> (!(Maybe SQLError), !(Maybe *MySQLCursor), !*MySQLConnection)
	sql_openCursor connection
		| connection == 0	= (Just (SQLProgrammingError 1 "Can not open a cursor on an uninitialized connection"),Nothing, connection) 
			# (cursor, connection) = sql_openCursorC connection
			| cursor == 0	= (Just (SQLInterfaceError 1 "Could not allocate memory for a new cursor"), Nothing, connection)
							= (Nothing, Just cursor, connection)
	where
		sql_openCursorC :: !*MySQLConnection -> (!*MySQLCursor,!*MySQLConnection);
		sql_openCursorC a0 = code {
			ccall sql_openCursorC "I:VII"
		}

	sql_closeCursor	:: !*MySQLCursor !*MySQLConnection -> (!(Maybe SQLError), !*MySQLConnection)
	sql_closeCursor cursor connection = (Nothing, sql_closeCursorC cursor connection)
	where
		sql_closeCursorC :: !*MySQLCursor !*MySQLConnection -> *MySQLConnection
		sql_closeCursorC a0 a1 = code {
			ccall sql_closeCursorC "I:V:I"
		}
/*
	//Database description (Optional)
	sql_tables			:: !*MySQLConnection							-> (!(Maybe SQLError), ![SQLTableName], !*MySQLConnection)
	sql_tables connection = (Just SQLNotSupportedError,[],connection)

	sql_describe		:: !SQLTableName !*MySQLConnection				-> (!(Maybe SQLError), !(Maybe SQLTable), !*MySQLConnection)
	sql_describe table connection = (Just SQLNotSupportedError, Nothing, connection)

	//Database management
	sql_createDatabase	:: !SQLDatabaseName !*MySQLConnection 			-> (!(Maybe SQLError), !*MySQLConnection)
	sql_createDatabase database connection = (Just SQLNotSupportedError, connection)

	sql_dropDatabase	:: !SQLDatabaseName !*MySQLConnection 			-> (!(Maybe SQLError), !*MySQLConnection)
	sql_dropDatabase database connection = (Just SQLNotSupportedError, connection)
*/
instance SQLCursor MySQLCursor
where
	sql_execute	:: !SQLStatement ![SQLValue] !*MySQLCursor -> (!(Maybe SQLError), !*MySQLCursor)
	sql_execute statement values cursor
		# (error, stmt, cursor)		= sql_execute_mkStatement statement values cursor
		| isJust error				= (error, cursor)
		# (error,cursor)			= sql_executeC stmt cursor
		| error <> 0				
			# (errno, errsize, cursor)	= sql_cursorErrorC cursor
			# errmsg					= createArray errsize '\0'
			# cursor					= sql_cursorErrorMessageC errmsg cursor
			= (Just (SQLDatabaseError errno errmsg), cursor)
		| otherwise					= (Nothing, cursor)
	where
		sql_execute_mkStatement :: !SQLStatement ![SQLValue] !*MySQLCursor -> (!(Maybe SQLError), !SQLStatement, !*MySQLCursor)
		sql_execute_mkStatement statement [] cursor
			# index			= sql_execute_markerIndex statement 0
			| index <> -1	= (Just (SQLProgrammingError 1 "Too many markers in SQL statement"), "", cursor)
							= (Nothing, statement, cursor)
		sql_execute_mkStatement statement [x:xs] cursor
			# index			= sql_execute_markerIndex statement 0
			| index == -1	= (Just (SQLProgrammingError 1 "Not enough markers in SQL statement"), "", cursor)
			# premarker							= statement % (0,index - 1)
			# postmarker						= statement % (index + 1, size statement - 1)
			# (x, cursor)						= sql_mkValueString x cursor
			# (xs_error,xs_statement, cursor)	= sql_execute_mkStatement postmarker xs cursor
			= (xs_error, premarker +++ x +++ xs_statement, cursor)

		//Find the index of the first '?' in the string (-1 if not found)
		sql_execute_markerIndex :: String Int -> Int
		sql_execute_markerIndex s i
			| i == size s		= -1
			| select s i == '?'	= i
			| otherwise			= sql_execute_markerIndex s (i + 1)

		sql_executeC :: !SQLStatement !*MySQLCursor -> (!Int, !*MySQLCursor)
		sql_executeC a0 a1 = code {
			ccall sql_executeC "SI:VII"
		}

	sql_executeMany :: !SQLStatement ![[SQLValue]] !*MySQLCursor -> (!(Maybe SQLError), !*MySQLCursor)
	sql_executeMany statement [] 	cursor = (Nothing, cursor)
	sql_executeMany statement [x:xs] cursor
		# (error, cursor)	= sql_execute statement x cursor
		| isJust error	= (error, cursor)
						= sql_executeMany statement xs cursor

	sql_insertId :: !*MySQLCursor -> (!(Maybe SQLError), !Int, !*MySQLCursor)
	sql_insertId cursor
		# (id, cursor) = sql_insertIdC cursor
		= (Nothing, id, cursor)
	where
		sql_insertIdC :: !*MySQLCursor -> (!Int, !*MySQLCursor)
		sql_insertIdC a0 = code {
			ccall sql_insertIdC "I:VII"
		}

	sql_numRows	:: !*MySQLCursor -> (!(Maybe SQLError), !Int, !*MySQLCursor)
	sql_numRows	cursor 
		# (num, cursor) = sql_numRowsC cursor
		= (Nothing, num, cursor)
	where
		sql_numRowsC :: !*MySQLCursor -> (!Int, !*MySQLCursor)
		sql_numRowsC a0 = code {
			ccall sql_numRowsC "I:VII"
		}

	sql_numFields :: !*MySQLCursor -> (!(Maybe SQLError), !Int, !*MySQLCursor)
	sql_numFields cursor
		# (num, cursor) = sql_numFieldsC cursor
		= (Nothing, num, cursor)
	where
		sql_numFieldsC :: !*MySQLCursor -> (!Int, !*MySQLCursor)
		sql_numFieldsC a0 = code {
			ccall sql_numFieldsC "I:VII"
		}

	sql_fetchOne :: !*MySQLCursor -> (!(Maybe SQLError), !(Maybe SQLRow), !*MySQLCursor)
	sql_fetchOne cursor
		# (_,num,cursor)	= sql_numFields cursor
		# (done,cursor)		= sql_fetchC cursor
		| done
			# (errno, errsize, cursor)	= sql_cursorErrorC cursor
			| errno == 0
				= (Nothing, Nothing, cursor)
			| otherwise
				# errmsg	= createArray errsize '\0'
				# cursor	= sql_cursorErrorMessageC errmsg cursor
				= (Just (SQLDatabaseError errno errmsg), Nothing, cursor)
		| otherwise	
			# (row, cursor)		= mapst sql_fetch_data [0..(num - 1)] cursor
			= (Nothing, Just row, cursor)
	where
		sql_fetch_data :: !Int !*MySQLCursor -> (!SQLValue, !*MySQLCursor)
		sql_fetch_data n cursor
			# (length, cursor)	= sql_fetchC_length n cursor
			# data				= createArray length '\0'
			# (type,cursor)		= sql_fetchC_data n data cursor //Dirty side-effect: data is updated destructively, but not returned.
			# data				= sql_fetch_mkvalue type data
			= (data,cursor)

		sql_fetch_mkvalue :: !Int !String -> SQLValue
		sql_fetch_mkvalue SQLTYPE_CHAR		s = SQLVChar s
		sql_fetch_mkvalue SQLTYPE_VARCHAR	s = SQLVVarchar s
		sql_fetch_mkvalue SQLTYPE_TEXT		s = SQLVText s
		sql_fetch_mkvalue SQLTYPE_INTEGER	s = SQLVInteger (toInt s)
		sql_fetch_mkvalue SQLTYPE_REAL		s = SQLVReal (toReal s)
		sql_fetch_mkvalue SQLTYPE_FLOAT		s = SQLVFloat (toReal s)
		sql_fetch_mkvalue SQLTYPE_DOUBLE	s = SQLVDouble (toReal s)
		sql_fetch_mkvalue SQLTYPE_DATE		s = SQLVDate {year = toInt (s % (0,3)), month = toInt (s % (5,6)), day = toInt (s % (8,9)), dayNr = 0}
		sql_fetch_mkvalue SQLTYPE_TIME		s = SQLVTime {hours = toInt (s % (0,1)), minutes = toInt (s % (3,4)), seconds = toInt (s % (6,7))}
		sql_fetch_mkvalue SQLTYPE_TIMESTAMP	s = SQLVTimestamp (toInt s)
		sql_fetch_mkvalue SQLTYPE_DATETIME	s = SQLVDatetime {year = toInt (s % (0,3)), month = toInt (s % (5,6)), day = toInt (s % (8,9)), dayNr = 0}
															 {hours = toInt (s % (11,12)), minutes = toInt (s % (14,15)), seconds = toInt (s % (17,18))}
		sql_fetch_mkvalue SQLTYPE_ENUM		s = SQLVEnum s
		sql_fetch_mkvalue SQLTYPE_NULL		s = SQLVNull
		sql_fetch_mkvalue _					s = SQLVUnknown s

		sql_fetchC :: !*MySQLCursor -> (!Bool,!*MySQLCursor);
		sql_fetchC a0 = code {
			ccall sql_fetchC "I:VII"
		}
		sql_fetchC_length :: !Int !*MySQLCursor -> (!Int,!*MySQLCursor);
		sql_fetchC_length a0 a1 = code {
			ccall sql_fetchC_length "II:VII"
		}
		sql_fetchC_data :: !Int !String !*MySQLCursor -> (!Int, !*MySQLCursor);
		sql_fetchC_data a0 a1 a2 = code {
			ccall sql_fetchC_data "ISI:VII"
		}

	sql_fetchMany :: !Int !*MySQLCursor -> (!(Maybe SQLError), ![SQLRow], !*MySQLCursor)
	sql_fetchMany 0 cursor = (Nothing, [], cursor)
	sql_fetchMany n cursor 
		# (error, row, cursor) = sql_fetchOne cursor
		= case row of	Nothing		= (Nothing, [], cursor)
						(Just x)
							# (error, xs, cursor) = sql_fetchMany (n - 1) cursor
							= (error, [x:xs], cursor)

	sql_fetchAll :: !*MySQLCursor -> (!(Maybe SQLError), ![SQLRow], !*MySQLCursor)
	sql_fetchAll cursor 
		# (error, row, cursor) = sql_fetchOne cursor
		= case row of	Nothing 	= (Nothing, [], cursor)
						(Just x)
							# (error, xs, cursor) = sql_fetchAll cursor
							= (error, [x:xs], cursor)

	sql_commit :: !*MySQLCursor -> (!(Maybe SQLError), !*MySQLCursor)
	sql_commit cursor = (Just SQLNotSupportedError, cursor)

	sql_rollback :: !*MySQLCursor -> (!(Maybe SQLError), !*MySQLCursor)
	sql_rollback cursor = (Just SQLNotSupportedError, cursor)
/*
	//Database management (Optional)
	sql_mkCreateTable	:: !SQLTable !*MySQLCursor -> (!SQLStatement, !*MySQLCursor)
	sql_mkCreateTable table cursor
		# (cols, cursor) = mapst mkCol table.SQLTable.columns cursor
		= ("CREATE TABLE " +++ table.SQLTable.name +++ " (\n" 
		+++ (foldr (+++) "" cols )
		+++ ")",cursor)
	where
		mkCol col cursor
			# (default, cursor) = mkDefault col.SQLColumn.default cursor
			# (type, cursor) = sql_mkTypeString col.SQLColumn.type cursor
			= ("\t" +++ col.SQLColumn.name +++ " "
			+++ type +++ " "
			+++ (if col.SQLColumn.null "NULL " "NOT NULL ")
			+++ default +++ "\n"
			  , cursor)
		mkDefault :: (Maybe SQLValue) *MySQLCursor -> (String,*MySQLCursor)
		mkDefault Nothing cursor = ("",cursor)
		mkDefault (Just val) cursor = sql_mkValueString val cursor

	sql_mkDropTable	:: !SQLTable !*MySQLCursor -> (!SQLStatement, !*MySQLCursor)
	sql_mkDropTable table cursor 
		= ("DROP TABLE " +++ table.SQLTable.name, cursor)

	sql_createTable	:: !SQLTable !*MySQLCursor -> (!(Maybe SQLError), !*MySQLCursor)
	sql_createTable table cursor 
		# (stmt, cursor) = sql_mkCreateTable table cursor
		= sql_execute stmt [] cursor

	sql_dropTable		:: !SQLTable !*MySQLCursor -> (!(Maybe SQLError), !*MySQLCursor)
	sql_dropTable table cursor 
		# (stmt, cursor) = sql_mkDropTable table cursor
		= sql_execute stmt [] cursor
*/
//Shared

//Convert an SQLValue to a string which is properly escaped for inclusion in an SQL statement
sql_mkValueString :: !SQLValue !*MySQLCursor -> (!String, !*MySQLCursor)
sql_mkValueString (SQLVChar s) cursor 
	# (s, cursor) = sql_escape s cursor
	= ("'" +++ s +++ "'", cursor)
sql_mkValueString (SQLVVarchar s) cursor 
	# (s, cursor) = sql_escape s cursor
	= ("'" +++ s +++ "'", cursor)
sql_mkValueString (SQLVText s) cursor 
	# (s, cursor) = sql_escape s cursor
	= ("'" +++ s +++ "'", cursor)
sql_mkValueString (SQLVInteger i) cursor = (toString i, cursor)
sql_mkValueString (SQLVReal r) cursor = (toString r, cursor)
sql_mkValueString (SQLVFloat f) cursor = (toString f, cursor)
sql_mkValueString (SQLVDouble d) cursor = (toString d, cursor)
sql_mkValueString (SQLVDate d) cursor
	= ("'" +++ (sql_formatDate d) +++  "'", cursor) 
sql_mkValueString (SQLVTime t) cursor
	= ("'" +++ (sql_formatTime t) +++ "'", cursor)
sql_mkValueString (SQLVTimestamp t) cursor = (toString t, cursor)
sql_mkValueString (SQLVDatetime d t) cursor 
	= ("'" +++ (sql_formatDate d)+++ " " +++ (sql_formatTime t) +++  "'", cursor) 
sql_mkValueString (SQLVEnum s) cursor
	# (s, cursor) = sql_escape s cursor
	= ("'" +++ s +++ "'", cursor)
sql_mkValueString (SQLVNull) cursor = ("NULL", cursor)
sql_mkValueString (SQLVUnknown s) cursor 
	# (s, cursor) = sql_escape s cursor
	= ("'" +++ s +++ "'", cursor)

//Escape a string value for insertion into the database.
//The cursor argument is required because it allows the escape function to take
//the character set of the database into account.
sql_escape :: !String !*MySQLCursor -> (!String, !*MySQLCursor)
sql_escape s cursor
	# escaped = createArray (2 * (size s) + 1 )	'\0'		//Create a buffer that is large enough to hold the escaped string
	# (escaped_size, cursor) = sql_escapeC s escaped cursor	//Do the real escaping in C (with dirty side effect on buffer)
	# escaped = escaped % (0, escaped_size - 1)				//Trim the buffer to the right size
	= (escaped, cursor)

sql_escapeC :: !SQLStatement !SQLStatement !*MySQLCursor -> (!Int,!*MySQLCursor);
sql_escapeC a0 a1 a2 = code {
	ccall sql_escapeC "SSI:VII"
}

sql_formatDate :: !Date -> String
sql_formatDate {year = year, month = month, day = day}
	=			(toString year) 
	+++ "-" +++ (toString month)
	+++ "-" +++	(toString day)

sql_formatTime :: !Time -> String
sql_formatTime {hours = hours, minutes = minutes, seconds = seconds}
	= 			(toString hours)
	+++ ":"	+++	(toString minutes)
	+++ ":" +++ (toString seconds)

//Convert an SQLType to a string which is properly escaped for inclusion in an SQL statement
sql_mkTypeString :: !SQLType !*MySQLCursor -> (!String, !*MySQLCursor)
sql_mkTypeString (SQLTChar i) cursor = ("CHAR (" +++ (toString i) +++ ")", cursor)
sql_mkTypeString (SQLTVarchar i) cursor = ("VARCHAR (" +++ (toString i) +++ ")", cursor)
sql_mkTypeString (SQLTText) cursor = ("TEXT", cursor)
sql_mkTypeString (SQLTInteger) cursor = ("INTEGER", cursor)
sql_mkTypeString (SQLTReal) cursor = ("REAL", cursor)
sql_mkTypeString (SQLTFloat) cursor = ("FLOAT", cursor)
sql_mkTypeString (SQLTDouble) cursor = ("DOUBLE", cursor)
sql_mkTypeString (SQLTDate) cursor = ("DATE", cursor)
sql_mkTypeString (SQLTTime) cursor = ("TIME", cursor)
sql_mkTypeString (SQLTTimestamp) cursor = ("TIMESTAMP",cursor)
sql_mkTypeString (SQLTDatetime) cursor = ("DATETIME",cursor)
sql_mkTypeString (SQLTEnum e) cursor 
	# (e, cursor) 	= mapst sql_mkValueString (map SQLVVarchar e) cursor
	= ("ENUM(" +++ (implode "," e) +++ ")",cursor)
sql_mkTypeString (SQLTUnknown i) cursor = ("TEXT",cursor)

//Utility
mapst :: (.a *st -> (.b, *st)) [.a] *st -> ([.b], *st)
mapst f [] st = ([],st)
mapst f [x:xs] st
	# (fx,st)	= f x st
	# (fxs,st)	= mapst f xs st
	= ([fx:fxs],st)

implode :: String [String] -> String
implode sep []	= ""
implode sep [x:[]] = x
implode sep [x:xs] = x +++ sep +++ (implode sep xs)
