implementation module GenSQL

import StdGeneric, StdMaybe
import SQL

import StdEnv
import Text

// Token type that is used in token lists with which reading and writing to
// a database can be viewed as parsing/unparsing of a stream of tokens
:: GSQLToken	= GSQLValue SQLValue				//Plain value, index in the stream determines to which field it is mapped
				| GSQLTerminator					//Terminator token, indicates the end of a list of values
				| GSQLOverride String SQLValue		//Field override. If there is such a token in the stream for a given field,
													//it's value is used instead of the 'normal' value in the stream

// Mode type which determines what we want the generic function to do
:: GSQLMode		= GSQLCreate	// C
				| GSQLRead		// R
				| GSQLUpdate	// U
				| GSQLDelete	// D
				| GSQLInfo		// Find info by traversing the structure of the type
				| GSQLInit		// Reduce a value to a flat list of tokens to start reading or deleting an entry.

// Information about the structure of types
:: GSQLFieldInfo =	{ fld_table		:: String			//The database table in which this value is stored
					, fld_select	:: Maybe String		//The database field in which this value is stored
					, fld_match		:: Maybe String		//The database field on which can be matched to find the right database record
					, rec_table		:: String			//The database table of the key field of the parent record
					, rec_key		:: String			//The database field of the key field of the parent record
					, val_list		:: Bool				//Are dealing with one, or with a set of values
					, val_maybe		:: Bool				//Is the value optional
					, val_fields	:: [GSQLFieldInfo]	//Information about the fields if this value is a record
					, val_id		:: Bool				//Is the field an ID type or an entity record
					}

// Some of the modes need to go in multiple passes over the data structure
:: GSQLPass		:== Int

// Error toString
instance toString GSQLError
where
	toString (GSQLTypeError e)		= "GSQLTypeError " +++ e
	toString (GSQLDatabaseError e)	= "GSQLDatabaseError " +++ toString e

//Wrapper functions
gsql_read :: !a !*cur -> (!(Maybe GSQLError), !(Maybe b), !*cur) | gSQL{|*|} a & gSQL{|*|} b & SQLCursor cur & bimap{|*|} cur
gsql_read id cursor
	# (mbErr,_,_,tokens,cursor)		= gSQL{|*|} GSQLInit 0 (Just id) [] [] cursor
	| isJust mbErr		= (mbErr, Nothing, cursor)
	# (mbErr,mbObj,_,tokens,cursor)	= gSQL{|*|} GSQLRead 0 Nothing [] tokens cursor
	| isEmpty tokens	= (mbErr, mbObj, cursor)
						= (mbErr, Nothing, cursor)
gsql_create	:: !b !*cur -> (!(Maybe GSQLError), !(Maybe a), !*cur) | gSQL{|*|} a & gSQL{|*|} b & SQLCursor cur & bimap{|*|} cur
gsql_create obj cursor
	# (mbErr,_,_,tokens,cursor)		= gSQL{|*|} GSQLCreate 0 (Just obj) [] [] cursor
	| isJust mbErr		= (mbErr, Nothing, cursor)
	# (mbErr,mbId,_,tokens,cursor)	= gSQL{|*|} GSQLRead 0 Nothing [] tokens cursor
	| isEmpty tokens	= (mbErr, mbId, cursor)
						= (mbErr, Nothing, cursor)

gsql_update :: !b !*cur -> (!(Maybe GSQLError), !(Maybe a), !*cur) | gSQL{|*|} a & gSQL{|*|} b & SQLCursor cur & bimap{|*|} cur
gsql_update obj cursor
	# (mbErr,_,_,tokens,cursor)		= gSQL{|*|} GSQLUpdate 0 (Just obj) [] [] cursor
	| isJust mbErr		= (mbErr, Nothing, cursor)
	# (mbErr,mbId,_,tokens,cursor)	= gSQL{|*|} GSQLRead 0 Nothing [] tokens cursor
	| isEmpty tokens	= (mbErr, mbId, cursor)
						= (mbErr, Nothing, cursor)

gsql_delete :: !a !*cur -> (!(Maybe GSQLError), !(Maybe b), !*cur) | gSQL{|*|} a & gSQL{|*|} b & SQLCursor cur & bimap{|*|} cur
gsql_delete id cursor
	# (mbErr,_,_,tokens,cursor)		= gSQL{|*|} GSQLInit 0 (Just id) [] [] cursor
	| isJust mbErr		= (mbErr, Nothing, cursor)
	# (mbErr,mbObj,_,tokens,cursor)	= gSQL{|*|} GSQLDelete 0 Nothing [] tokens cursor
	| isEmpty tokens	= (mbErr, mbObj, cursor)
						= (mbErr, Nothing, cursor)

//The real generic magic!
generic	gSQL t :: !GSQLMode !GSQLPass !(Maybe t) ![GSQLFieldInfo] ![GSQLToken] !*cur -> (!(Maybe GSQLError), !(Maybe t), ![GSQLFieldInfo], ![GSQLToken], !*cur) | SQLCursor cur

gSQL{|Int|} 	GSQLInfo	_	_			info	tokens							cursor = (Nothing, Nothing, [emptyInfo], tokens, cursor)
gSQL{|Int|} 	GSQLRead	_	_			info	[GSQLValue (SQLVInteger x):xs]	cursor = (Nothing, Just x, info, xs, cursor)
gSQL{|Int|} 	GSQLInit	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, [GSQLValue (SQLVInteger x):tokens], cursor)
gSQL{|Int|} 	GSQLCreate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVInteger x)], cursor)
gSQL{|Int|} 	GSQLUpdate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVInteger x)], cursor)
gSQL{|Int|} 	GSQLDelete	_	_			info	[GSQLValue (SQLVInteger x):xs]	cursor = (Nothing, Just x, info, xs, cursor)
gSQL{|Int|}		_			_	_			info	tokens							cursor = (Nothing, Nothing, info, tokens, cursor)

gSQL{|Real|}	GSQLInfo	_	_			info	tokens							cursor = (Nothing, Nothing, [emptyInfo], tokens, cursor)
gSQL{|Real|}	GSQLRead	_	_			info	[GSQLValue (SQLVReal x):xs]		cursor = (Nothing, Just x, info, xs, cursor)
gSQL{|Real|}	GSQLInit	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, [GSQLValue (SQLVReal x):tokens], cursor)
gSQL{|Real|}	GSQLCreate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVReal x)], cursor)
gSQL{|Real|}	GSQLUpdate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVReal x)], cursor)
gSQL{|Real|}	GSQLDelete	_	_			info	[GSQLValue (SQLVReal x):xs]		cursor = (Nothing, Just x, info, xs, cursor)
gSQL{|Real|}	_			_	_			info	tokens							cursor = (Nothing, Nothing, info, tokens, cursor)

gSQL{|Bool|} 	GSQLInfo	_	_			info	tokens							cursor = (Nothing, Nothing, [emptyInfo], tokens, cursor)
gSQL{|Bool|} 	GSQLRead	_	_			info	[GSQLValue (SQLVInteger x):xs]	cursor = (Nothing, Just (x <> 0), info, xs, cursor)
gSQL{|Bool|} 	GSQLInit	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, [GSQLValue (SQLVInteger (if x 1 0)):tokens], cursor)
gSQL{|Bool|} 	GSQLCreate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVInteger (if x 1 0))], cursor)
gSQL{|Bool|} 	GSQLUpdate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVInteger (if x 1 0))], cursor)
gSQL{|Bool|} 	GSQLDelete	_	_			info	[GSQLValue (SQLVInteger x):xs]	cursor = (Nothing, Just (x <> 0), info, xs, cursor)
gSQL{|Bool|} 	_			_	_			info	tokens							cursor = (Nothing, Nothing, info, tokens, cursor)

gSQL{|Char|} 	GSQLInfo	_	_			info	tokens							cursor = (Nothing, Nothing, [emptyInfo], tokens, cursor)
gSQL{|Char|} 	GSQLRead	_	_			info	[GSQLValue (SQLVChar x):xs]		cursor = (Nothing, Just (hd (fromString x)), info, xs, cursor)
gSQL{|Char|} 	GSQLInit	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, [GSQLValue (SQLVChar (toString [x])):tokens], cursor)
gSQL{|Char|} 	GSQLCreate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVChar (toString [x]))], cursor)
gSQL{|Char|} 	GSQLUpdate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVChar (toString [x]))], cursor)
gSQL{|Char|} 	GSQLDelete	_	_			info	[GSQLValue (SQLVChar x):xs]		cursor = (Nothing, Just (hd (fromString x)), info, xs, cursor)
gSQL{|Char|} 	_			_	_			info	tokens							cursor = (Nothing, Nothing, info, tokens, cursor)

gSQL{|String|} 	GSQLInfo	_	_			info	tokens							cursor = (Nothing, Nothing, [emptyInfo], tokens, cursor)
gSQL{|String|}	GSQLRead	_	_			info	[GSQLValue (SQLVVarchar x):xs]	cursor = (Nothing, Just x, info, xs, cursor)
gSQL{|String|}	GSQLInit	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, [GSQLValue (SQLVVarchar x):tokens], cursor)
gSQL{|String|}	GSQLCreate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVVarchar x)], cursor)
gSQL{|String|}	GSQLUpdate	_	(Just x)	info	tokens							cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue (SQLVVarchar x)], cursor)
gSQL{|String|}	GSQLDelete	_	_			info	[GSQLValue (SQLVVarchar x):xs]	cursor = (Nothing, Just x, info, xs, cursor)
gSQL{|String|}	_			_	_			info	tokens							cursor = (Nothing, Nothing, info, tokens, cursor)

gSQL{|UNIT|}	GSQLInfo	_	_			info	tokens							cursor = (Nothing, Nothing, [emptyInfo], tokens, cursor)
gSQL{|UNIT|}	GSQLRead	_	_			info	tokens							cursor = (Nothing, Just UNIT, info, tokens, cursor)
gSQL{|UNIT|}	GSQLDelete	_	_			info	tokens							cursor = (Nothing, Just UNIT, info, tokens, cursor)
gSQL{|UNIT|}	_			_	_			info	tokens							cursor = (Nothing, Nothing, info, tokens, cursor)

//Default function for EITHER 
gSQL{|EITHER|} fl fr mode pass Nothing info tokens cursor
	= case fl mode pass Nothing info tokens cursor of
		(mbErr,Just x, ixs, xs, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Just (LEFT x), ixs, xs, cursor) 					
		(mbErr,Nothing, _, _, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= case fr mode pass Nothing info tokens cursor of
				(mbErr,Just x, ixs, xs, cursor)
					| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
					= (Nothing, Just (RIGHT x), ixs, xs, cursor)
				(mbErr,Nothing, _,  _, cursor)
					| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
					= (Nothing, Nothing, info, tokens, cursor)

gSQL{|EITHER|} fl fr mode pass (Just (LEFT x)) info tokens cursor
	# (mbErr,_,info,tokens,cursor) = fl mode pass (Just x) info tokens cursor
	| isJust mbErr	= (mbErr,Nothing,[],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)
gSQL{|EITHER|} fl fr mode pass (Just (RIGHT x)) info tokens cursor
	# (mbErr,_,info,tokens,cursor) = fr mode pass (Just x) info tokens cursor
	| isJust mbErr	= (mbErr,Nothing,[],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

//Special PAIR for GSQLInfo mode. Just combines the two info lists
gSQL{|PAIR|} fx fy GSQLInfo _ _ info tokens cursor
	# (mbErr,_, infox, _, cursor)	= fx GSQLInfo 0 Nothing info tokens cursor
	| isJust mbErr					= (mbErr,Nothing,[],[],cursor)
	# (mbErr,_, infoy, _, cursor)	= fy GSQLInfo 0 Nothing info tokens cursor
	| isJust mbErr					= (mbErr,Nothing,[],[],cursor)
	= (Nothing, Nothing, infox ++ infoy, tokens, cursor)

//Default function for PAIR
gSQL{|PAIR|} fx fy mode pass Nothing info tokens cursor
	# (mbErr, resx, infox, tokx, cursor)	= fx mode pass Nothing info tokens cursor
	| isJust mbErr							= (mbErr,Nothing,[],[],cursor)
	# (mbErr, resy, infoy, toky, cursor)	= fy mode pass Nothing infox tokx cursor
	| isJust mbErr							= (mbErr,Nothing,[],[],cursor)
	| isJust resx && isJust resy			= (Nothing, Just (PAIR (fromJust resx) (fromJust resy)), infoy, toky, cursor)
											= (Nothing, Nothing, infoy, toky, cursor)

gSQL{|PAIR|} fx fy mode pass (Just (PAIR x y)) info tokens cursor
	# (mbErr,_, infox, tokx, cursor)	= fx mode pass (Just x) info tokens cursor
	| isJust mbErr						= (mbErr,Nothing,[],[],cursor)
	# (mbErr,_, infoy, toky, cursor)	= fy mode pass (Just y) infox tokx cursor
	| isJust mbErr						= (mbErr,Nothing,[],[],cursor)
	= (Nothing, Nothing, infoy, toky, cursor)

//Special CONS for GSQLInfo mode.
gSQL{|CONS of d|} f GSQLInfo _ _ info tokens cursor
	| not (isEmpty d.gcd_fields)
		= (Nothing, Nothing, [{emptyInfo & val_fields = [setFldInfo desc emptyInfo \\ desc <- d.gcd_fields] , val_id = isID d.gcd_name }], tokens, cursor)
	| otherwise
		# (mbErr, _, info, _, cursor)		= f GSQLInfo 0 Nothing info tokens cursor
		| isJust mbErr						= (mbErr,Nothing,[],[],cursor)
		= (Nothing, Nothing, info, tokens, cursor)

//Special CONS for GSQLRead mode.
gSQL{|CONS of d|} f GSQLRead _ _ info tokens cursor
	| not (isEmpty d.gcd_fields)
		# (_,_,info,_,cursor)	= f GSQLInfo 0 Nothing info tokens cursor							//Extract info about the fields in the record
		# info					= map (setRecInfo info) info										//Add type info about this record
		# (mbErr,tokx,cursor)	= readRecord info (hd tokens) cursor								//Read the extra tokens for this record
		| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
		# tokens				= tokx ++ (tl tokens)												//Replace the head of the token list by the extra tokens
		= case f GSQLRead 0 Nothing info tokens cursor of
			(mbErr, Nothing, _, _, cursor)
				| isJust mbErr	= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Nothing, info, tokens, cursor)
			(mbErr, Just x, _, xs, cursor)
				| isJust mbErr	= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Just (CONS x), info, xs, cursor)
	| otherwise
		= case f GSQLRead 0 Nothing info tokens cursor of
			(mbErr, Nothing, _, _, cursor)
				| isJust mbErr	= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Nothing, info, tokens, cursor)
			(mbErr,Just x, _, xs, cursor)
				| isJust mbErr	= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Just (CONS x), info, xs, cursor)

//Special CONS for GSQLCreate mode.
gSQL{|CONS of d|} f GSQLCreate pass (Just (CONS x)) info tokens cursor 
	| not (isEmpty d.gcd_fields) && not (isID d.gcd_name)
		# (mbErr,_,info,_,cursor)	= f GSQLInfo 0 (Just x) info [] cursor							//Find type info of the individual fields
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# info						= map (setRecInfo info) info 									//Add type info about this record
		# (overrides,tokens)		= takeOverrides tokens
		# (mbErr,_,_,tokx1,cursor)	= f GSQLCreate 1 (Just x) info [] cursor						//First pass: Prepare the tokenstream
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (mbErr,id,cursor)			= insertRecord info tokx1 overrides cursor						//Write the values in this record to the database
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (mbErr,_,_,tokx2,cursor)	= f GSQLCreate 2 (Just x) info (mkOverrides info id) cursor		//Second pass: Recursive create for incoming pointers
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (_,tokx2)					= takeOverrides tokx2											//Remove the overrides used in the second pass
		# tokx						= mergeTokens info tokx1 tokx2									//Merge the tokens from both passes
		# (mbErr,cursor)			= linkRecords info tokx id cursor								//Add the link records
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, info, overrides ++ tokens ++ [id], cursor)
	| otherwise
		# (mbErr,_,_,tokens,cursor) = f GSQLCreate pass (Just x) info tokens cursor
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, info, tokens, cursor)

//Special CONS for GSQLUpdate mode.
gSQL{|CONS of d|} f GSQLUpdate pass (Just (CONS x)) info tokens cursor
	| not (isEmpty d.gcd_fields) && not (isID d.gcd_name)
		# (mbErr,_,info,_,cursor)	= f GSQLInfo 0 (Just x) info [] cursor							//Find type info of the individual fields
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# info						= map (setRecInfo info) info									//Add type info about this record
		# (overrides,tokens)		= takeOverrides tokens
		# (mbErr,_,_,tokx1,cursor)	= f GSQLUpdate 1 (Just x) info [] cursor						//First pass: Prepare the tokenstream
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (mbErr,orig,cursor)		= readRecord info (hd tokx1) cursor								//Read the original values
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (mbErr,id,cursor)			= updateOrInsertRecord info tokx1 overrides cursor				//Update the values in this record in the database
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (mbErr,_,_,tokx2,cursor)	= f GSQLUpdate 2 (Just x) info (mkOverrides info id) cursor		//Second pass: Recursive update for incoming pointers
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (_,tokx2)					= takeOverrides tokx2											//Remove the overrides used in the second pass
		# tokx						= mergeTokens info tokx1 tokx2									//Merge the tokens from both passes
		# (mbErr,cursor)			= relinkRecords info tokx id cursor								//Update, or add the link records
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# tokx						= zipTokens info orig tokx										//Zip the original values with the updated ones
		# (mbErr,cursor)			= unlinkDirectRecords info tokx cursor
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# (mbErr,_,_,_,cursor)		= f GSQLUpdate 3 (Just x) info tokx cursor						//Garbage collect, delete removed entities in the database
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		= (Nothing,Nothing, info, overrides ++ tokens ++ [id], cursor)
	| otherwise
		# (mbErr,_,_,tokens,cursor) = f GSQLUpdate pass (Just x) info tokens cursor
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, info, tokens, cursor)

//Special CONS for GSQLDelete mode.
gSQL{|CONS of d|} f GSQLDelete pass _ info tokens cursor
	| not (isEmpty d.gcd_fields) && not (isID d.gcd_name)
		# (mbErr,_,info,_,cursor)	= f GSQLInfo 0 Nothing [] tokens cursor							//Extract info about the fields in the record
		| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
		# info						= map (setRecInfo info) info									//Add type info about this record
		# id						= hd tokens
		| pass == 0 //Read and delete
			# (mbErr,tokx,cursor)	= readRecord info id cursor										//Read the extra tokens for this record
			| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
			# tokens				= tokx ++ (tl tokens)											//Replace the head of the token list by the extra tokens
			# (mbErr,cursor)		= unlinkRecords info id cursor									//Unlink indirect pointed fields
			| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
			= case f GSQLDelete 0 Nothing info tokens cursor of
				(mbErr, Nothing, _, _, cursor)
						| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
						= (Nothing, Nothing, info, tokens, cursor)
				(mbErr, Just x, _, xs, cursor)
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						# (mbErr,cursor)		= deleteRecord info id cursor
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						# (mbErr,_,_,_,cursor)	= f GSQLDelete 2 Nothing info tokens cursor			//Garbage collect
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						= (Nothing, Just (CONS x), info, xs, cursor)
		| pass == 1 //Only read
			# (mbErr,tokx,cursor)	= readRecord info id cursor										//Read the extra tokens for this record
			| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
			# tokens				= tokx ++ (tl tokens)											//Replace the head of the token list by the extra tokens
			# (mbErr,cursor)		= unlinkRecords info id cursor									//Unlink indirect pointed fields
			| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
			= case f GSQLDelete 0 Nothing info tokens cursor of
				(mbErr, Nothing, _, _, cursor)
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						= (Nothing, Nothing, info, tokens, cursor)
				(mbErr, Just x, _, xs, cursor)
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						# (mbErr,_,_,_,cursor)	= f GSQLDelete 2 Nothing info tokens cursor			//Garbage collect
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						= (Nothing, Just (CONS x), info, xs, cursor)
		| pass == 2 //Garbage collection: construct and delete
			= case f GSQLDelete 1 Nothing info tokens cursor of
				(mbErr, Nothing, _, _, cursor)
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						= (Nothing, Nothing, info, tokens, cursor)
				(mbErr, Just x, _, xs, cursor)
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						# (mbErr,cursor)		= deleteRecord info id cursor
						| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
						= (Nothing, Just (CONS x), info, xs, cursor)
		| otherwise
			= (Nothing, Nothing, info, tokens, cursor)
	| otherwise
		= case f GSQLDelete pass Nothing info tokens cursor of
			(mbErr, Nothing, _, _, cursor)
				| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Nothing, info, tokens, cursor)
			(mbErr, Just x, _, xs, cursor)
				| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Just (CONS x), info, xs, cursor)

//Default function for CONS
gSQL{|CONS|} f mode pass Nothing info tokens cursor
	= case f mode pass Nothing info tokens cursor of
		(mbErr, Nothing, _, _, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)
		(mbErr, Just x, ixs, xs, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Just (CONS x), ixs, xs, cursor)

gSQL{|CONS|} f mode pass (Just (CONS x)) info tokens cursor
	# (mbErr,_,info,tokens,cursor)	= f mode pass (Just x) info tokens cursor
	| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

//Special FIELD for GSQLInfo mode.
gSQL{|FIELD of d|} f GSQLInfo _ _ info tokens cursor 
	= case f GSQLInfo 0 Nothing info tokens cursor of
		(mbErr,_, [x:xs], _, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, [setFldInfo d x:xs], tokens, cursor)
		(mbErr,_, info, _, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)

//Special FIELD for GSQLInit mode.
gSQL{|FIELD of d|} f GSQLInit _ (Just (FIELD x)) info tokens cursor
	| d.gfd_index == 0
		# (mbErr,_,info,tokens,cursor)	= f GSQLInit 0 (Just x) info tokens cursor
		| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, info, tokens, cursor)
	| otherwise
		= (Nothing, Nothing, info, tokens, cursor)

//Special FIELD for GSQLCreate mode.
gSQL{|FIELD|} f GSQLCreate 1 (Just (FIELD x)) [i:is] tokens cursor
	| store i	
		# (mbErr,_,_,tokens,cursor)		= f GSQLCreate 1 (Just x) [] tokens cursor
		| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, is, tokens, cursor)
	| otherwise
		= (Nothing, Nothing, is, tokens, cursor)

gSQL{|FIELD|} f GSQLCreate 2 (Just (FIELD x)) [i:is] tokens cursor
	| store i
		= (Nothing, Nothing, is, tokens, cursor)
	| otherwise
		# (mbErr,_,_,tokens,cursor)		= f GSQLCreate 2 (Just x) [] tokens cursor
		| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, is, tokens, cursor)

//Special FIELD for GSQLUpdate mode.
gSQL{|FIELD|} f GSQLUpdate 1 (Just (FIELD x)) [i:is] tokens cursor
	| store i	
		# (mbErr,_,_,tokens,cursor)		= f GSQLUpdate 1 (Just x) [] tokens cursor
		| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, is, tokens, cursor)
	| otherwise
		= (Nothing, Nothing, is, tokens, cursor)

gSQL{|FIELD|} f GSQLUpdate 2 (Just (FIELD x)) [i:is] tokens cursor
	| store i
		= (Nothing, Nothing, is, tokens, cursor)
	| otherwise
		# (mbErr,_,_,tokens,cursor)		= f GSQLUpdate 2 (Just x) [] tokens cursor
		| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, is, tokens, cursor)

gSQL{|FIELD|} f GSQLUpdate 3 (Just (FIELD x)) [i:is] tokens cursor
	# (orig,tokens)	= takeTokens i.val_list tokens
	# (new,tokens)	= takeTokens i.val_list tokens
	| i.val_list
		# (mbErr,_,_,_,cursor)			= f GSQLDelete 0 Nothing [] (removedTokens orig new) cursor 
		| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
		= (Nothing, Nothing, is, tokens, cursor)
	| i.val_maybe
		| (isNullValue (hd new)) && (not (isNullValue (hd orig)))
			# (mbErr,_,_,_,cursor)		= f GSQLDelete 0 Nothing [] orig cursor 
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, is, tokens, cursor)
		| otherwise
			= (Nothing, Nothing, is, tokens, cursor)
	| otherwise
		= (Nothing, Nothing, is, tokens, cursor)

//Special FIELD for GSQLDelete mode.
gSQL{|FIELD of d|} f GSQLDelete 0 _ [i:is] tokens cursor
	| isOutPointer i
		= case f GSQLDelete 1 Nothing [] tokens cursor of
			(mbErr,Nothing,_,_,cursor)
				| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Nothing, is, tokens, cursor)
			(mbErr,Just x,_,xs,cursor)
				| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Just (FIELD x), is, xs, cursor)
	| otherwise
		= case f GSQLDelete 0 Nothing [] tokens cursor of
			(mbErr,Nothing,_,_,cursor)
				| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Nothing, is, tokens, cursor)
			(mbErr,Just x,_,xs,cursor)
				| isJust mbErr			= (mbErr,Nothing, [],[],cursor)
				= (Nothing, Just (FIELD x), is, xs, cursor)

gSQL{|FIELD of d|} f GSQLDelete pass _ [i:is] tokens cursor
	= case f GSQLDelete pass Nothing [] tokens cursor of
		(mbErr,Nothing,_,_,cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, is, tokens, cursor)
		(mbErr,Just x,_,xs,cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Just (FIELD x), is, xs, cursor)

//Default function for FIELD
gSQL{|FIELD|} f mode pass Nothing info tokens cursor 
	= case f mode pass Nothing info tokens cursor of
		(mbErr,Nothing, _, _, cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)
		(mbErr,Just x, ixs, xs, cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Just (FIELD x), ixs, xs, cursor)

gSQL{|FIELD|} f mode pass (Just (FIELD x)) info tokens cursor
	# (mbErr,_, info, tokens, cursor)	= f mode pass (Just x) info tokens cursor
	| isJust mbErr						= (mbErr,Nothing, [],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

//Special OBJECT for GSQLInfo mode.
gSQL{|OBJECT|} f GSQLInfo pass Nothing info tokens cursor
	# (mbErr,_, info, _, cursor)		= f GSQLInfo 0 Nothing info tokens cursor
	| isJust mbErr						= (mbErr,Nothing, [],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

//Default function for OBJECT
gSQL{|OBJECT|} f mode pass Nothing info tokens cursor 
	# (mbErr, res, info, tokens, cursor)	= f mode pass Nothing info tokens cursor
	| isJust mbErr							= (mbErr,Nothing, [],[],cursor)
	| isJust res							= (Nothing, Just (OBJECT (fromJust res)), info, tokens, cursor)
											= (Nothing, Nothing, info, tokens, cursor)

gSQL{|OBJECT|} f mode pass (Just (OBJECT x)) info tokens cursor
	# (mbErr,_, info, tokens, cursor)		= f mode pass (Just x) info tokens cursor
	| isJust mbErr							= (mbErr,Nothing, [],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

//Bullocks case, just because the compiler doesn't see this will never happen
gSQL{|OBJECT|} _ _ _ _ info tokens cursor = (Nothing, Nothing, info, tokens, cursor)

//We don't use arrays in the representation types
gSQL{|{}|} f mode pass mb_val info tokens cursor = (Just (GSQLTypeError "Arrays are not representation types"), Nothing, info, tokens, cursor)
gSQL{|{!}|} f mode pass mb_val info tokens cursor = (Just (GSQLTypeError "Arrays are not representation types"), Nothing, info, tokens, cursor)

//The maybe type is mapped to NULL values in the database
//A NULL value maps to Nothing. A non-NULL value to a Just.
gSQL{|Maybe|} f GSQLRead _ _ info [GSQLValue (SQLVNull):xs] cursor = (Nothing, Just Nothing, info, xs, cursor)
gSQL{|Maybe|} f GSQLRead _ _ info tokens cursor
	= case f GSQLRead 0 Nothing info tokens cursor of
		(mbErr, Nothing, _, _, cursor)
			| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)
		(mbErr, Just x, _, xs, cursor)
			| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Just (Just x), info, xs, cursor)

gSQL{|Maybe|} f GSQLInfo _ _ info tokens cursor
	= case f GSQLInfo 0 Nothing info tokens cursor of
		(mbErr,_, [x:xs], _, cursor)
			| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, [{x & val_maybe = True}:xs], tokens, cursor)
		(mbErr,_, info, _, cursor)
			| isJust mbErr					= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)

gSQL{|Maybe|} f GSQLInit _ (Just Nothing) info tokens cursor = (Nothing, Nothing, info, [GSQLValue SQLVNull:tokens], cursor)
gSQL{|Maybe|} f GSQLInit _ (Just (Just x)) info tokens cursor
	# (mbErr, _,info, tokens, cursor)	= f GSQLInit 0 (Just x) info tokens cursor
	| isJust mbErr						= (mbErr,Nothing, [],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

gSQL{|Maybe|} f GSQLCreate _ (Just Nothing) info tokens cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue SQLVNull], cursor)
gSQL{|Maybe|} f GSQLCreate _ (Just (Just x)) info tokens cursor
	# (mbErr, _,_, tokens, cursor)		= f GSQLCreate 0 (Just x) info tokens cursor
	| isJust mbErr						= (mbErr,Nothing, [],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

gSQL{|Maybe|} f GSQLUpdate _ (Just Nothing) info tokens cursor = (Nothing, Nothing, info, tokens ++ [GSQLValue SQLVNull], cursor)
gSQL{|Maybe|} f GSQLUpdate _ (Just (Just x)) info tokens cursor
	# (mbErr,_,_, tokens, cursor)		= f GSQLUpdate 0 (Just x) info tokens cursor
	| isJust mbErr						= (mbErr,Nothing, [],[],cursor)
	= (Nothing, Nothing, info, tokens, cursor)

gSQL{|Maybe|} f GSQLDelete _ _ info [GSQLValue (SQLVNull):xs] cursor = (Nothing, Just Nothing, info, xs, cursor)
gSQL{|Maybe|} f GSQLDelete pass _ info tokens cursor
	= case f GSQLDelete pass Nothing info tokens cursor of
		(mbErr, Nothing, _, _, cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)
		(mbErr, Just x, _, xs, cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Just (Just x), info, xs, cursor)

gSQL{|Maybe|} f  _ _ _ info tokens cursor = (Nothing, Nothing, info, tokens, cursor)

//The list type is used for one-to-many and many-to-many relations
//in the database. To construct a list of values, it expects the values
//concatenated in the stream followed by a terminator token.
gSQL{|[]|} f GSQLRead _ _ info [GSQLTerminator:xs] cursor = (Nothing, Just [], info, xs, cursor)
gSQL{|[]|} f GSQLRead _ _ info tokens cursor
	= case f GSQLRead 0 Nothing info tokens cursor of
		(mbErr,Nothing, _, _, cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)
		(mbErr,Just x, _, xs, cursor)
			| isJust mbErr				= (mbErr,Nothing, [],[],cursor)
			= case gSQL{|*->*|} f GSQLRead 0 Nothing info xs cursor of
				(mbErr, Nothing, _, _, cursor)
					| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
					= (Nothing, Nothing, info, tokens, cursor)
				(mbErr, Just y, _, ys, cursor)
					| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
					= (Nothing, Just [x:y], info, ys, cursor)

//Info case sets the list flag to true
gSQL{|[]|} f GSQLInfo _ _ info tokens cursor
	= case f GSQLInfo 0 Nothing info tokens cursor of
		(mbErr, _, [x:xs], _, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, [{x & val_list = True}:xs], tokens, cursor)
		(mbErr, _, info, _, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)


//Create case does create for each member of the list and adds a terminator token 
//to the token stream.
gSQL{|[]|} f GSQLCreate _ (Just list) info tokens cursor
	# (mbErr, tokens, cursor) = doList list tokens cursor
	= (mbErr, Nothing, info, tokens, cursor)
where
	doList [] tokens cursor = (Nothing, tokens ++ [GSQLTerminator], cursor)
	doList [x:xs] tokens cursor
		# (mbErr, _, _, tokx, cursor)	= f GSQLCreate 0 (Just x) [] tokens cursor
		| isJust mbErr		= (mbErr,[],cursor)
		= doList xs tokx cursor

//Update case
gSQL{|[]|} f GSQLUpdate _ (Just list) info tokens cursor
	# (mbErr, tokens, cursor) = doList list tokens cursor
	= (mbErr, Nothing, info, tokens, cursor)
where
	doList [] tokens cursor = (Nothing, tokens ++ [GSQLTerminator], cursor)
	doList [x:xs] tokens cursor
		# (mbErr, _, _, tokx, cursor)	= f GSQLUpdate 0 (Just x) [] tokens cursor
		| isJust mbErr		= (mbErr,[],cursor)
		= doList xs tokx cursor

//Delete case
gSQL{|[]|} f GSQLDelete _ _ info [GSQLTerminator:xs] cursor = (Nothing, Just [], info, xs, cursor)
gSQL{|[]|} f GSQLDelete pass _ info tokens cursor
	= case f GSQLDelete pass Nothing info tokens cursor of
		(mbErr, Nothing, _, _, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= (Nothing, Nothing, info, tokens, cursor)
		(mbErr, Just x, _, xs, cursor)
			| isJust mbErr		= (mbErr,Nothing, [],[],cursor)
			= case gSQL{|*->*|} f GSQLDelete pass Nothing info xs cursor of
				(mbErr, Nothing, _, _, cursor)
					| isJust mbErr	= (mbErr,Nothing, [],[],cursor)
					= (Nothing, Nothing, info, tokens, cursor)
				(mbErr, Just y, _, ys, cursor)
					| isJust mbErr	= (mbErr,Nothing, [],[],cursor)
					= (Nothing, Just [x:y], info, ys, cursor)

//Default function for lists
gSQL{|[]|} f mode pass (Just list) info tokens cursor
	# (mbErr, tokens, cursor) = doList mode pass list tokens cursor
	= (mbErr, Nothing, info, tokens, cursor)
where
	doList mode pass [] tokens cursor = (Nothing, [GSQLTerminator], cursor)
	doList mode pass [x:xs] tokens cursor
		# (mbErr, _,info, tokx, cursor)	= f mode pass (Just x) info tokens cursor
		| isJust mbErr	= (mbErr, [], cursor)
		# (mbErr, tokxs, cursor)		= doList mode pass xs tokens cursor
		= (mbErr, tokx ++ tokxs, cursor)

gSQL{|[]|} f  _ _ _ info tokens cursor = (Nothing, Nothing, info, tokens, cursor)

// --- Helper functions for GSQLInfo mode --- //

//Creates a default info record
emptyInfo :: GSQLFieldInfo 
emptyInfo = {fld_table = undef, fld_select = Nothing, fld_match = Nothing, rec_table = undef, rec_key = undef, val_list = False, val_maybe = False, val_fields = [], val_id = False}

//Sets the information that is encoded in the field name in an info record
setFldInfo :: !GenericFieldDescriptor !GSQLFieldInfo -> GSQLFieldInfo
setFldInfo desc info 
	# parts = text_split gsql_fieldSeparator desc.gfd_name
	//Determine the table
	# table = hd parts
	# parts = tl parts
	//Determine the select columns
	# select = takeWhile ((<>) "ofwhich") parts		//Collect all elements before "ofwhich"
	# select = if (isEmpty select) Nothing (Just (hd select))
	# match = (dropWhile ((<>) "ofwhich") parts)	//Collect all elements after and including "ofwhich"
	# match = if (isEmpty match) match (tl match)	//Strip the "ofwhich" element from the select list
	# match = if (isEmpty match) Nothing (Just (hd match))
	= {info & fld_table = table, fld_select = select, fld_match = match}

//Copies the table and key from the first field in a set to the info record
setRecInfo :: ![GSQLFieldInfo] !GSQLFieldInfo -> GSQLFieldInfo
setRecInfo fields info = {info & rec_table = getTable fields, rec_key = getKey fields}

// --- Helper functions for GSQLRead mode --- //

readRecord :: [GSQLFieldInfo] GSQLToken !*cur -> (Maybe GSQLError, [GSQLToken], *cur) | SQLCursor cur
readRecord [] id cursor = (Nothing, [], cursor)
readRecord [i:is] id cursor
	# (mbErr, cursor)			= sql_execute (mkSelectQuery i) [fromValue id] cursor 
	| isJust mbErr				= (Just (GSQLDatabaseError (fromJust mbErr)), [], cursor)
	# (mbErr, rows, cursor)		= sql_fetchAll cursor
	| isJust mbErr				= (Just (GSQLDatabaseError (fromJust mbErr)), [], cursor)
	#  tokx						= map GSQLValue (flatten rows)
	# (mbErr,tokxs,cursor)		= readRecord is id cursor
	| isJust mbErr				= (mbErr, [], cursor)
	| i.val_list	= (Nothing, tokx ++ [GSQLTerminator] ++ tokxs, cursor)
					= (Nothing, tokx ++ tokxs, cursor)

// --- Helper functions for GSQLCreate mode --- //

insertRecord :: [GSQLFieldInfo] [GSQLToken] [GSQLToken] *cur -> (Maybe GSQLError, GSQLToken,*cur) | SQLCursor cur
insertRecord info values overrides cursor 
	# info				= filter store info 
	# stmt				= mkInsertQuery info
	# values			= appOverrides info values overrides
	# keyval			= hd values
	# (mbErr,cursor)	= sql_execute stmt (map fromValue values) cursor
	| isJust mbErr		= (Just (GSQLDatabaseError (fromJust mbErr)), keyval, cursor)
	| isZeroValue keyval	
		# (mbErr, idval, cursor) = fetchIDToken cursor
		| isJust mbErr		= (Just (GSQLDatabaseError (fromJust mbErr)), keyval, cursor)
		= (Nothing, idval, cursor)
	| otherwise
		= (Nothing, keyval, cursor)

//Checks the link records and adds new links where necessary
linkRecords :: [GSQLFieldInfo] [GSQLToken] GSQLToken !*cur -> (Maybe GSQLError, *cur ) | SQLCursor cur
linkRecords [] tokens id cursor = (Nothing, cursor)
linkRecords [i:is] tokens id cursor
	# (values,tokens)	= takeTokens i.val_list tokens
	| isIndirect i
		# (mbErr, cursor) = linkRecord i (map fromValue (init values)) (fromValue id) cursor
		| isJust mbErr = (mbErr, cursor)
		= linkRecords is tokens id cursor
	| otherwise
		= linkRecords is tokens id cursor
where
	linkRecord info [] id cursor = (Nothing, cursor)
	linkRecord info [x:xs] id cursor
		# (mbErr, cursor)	= sql_execute (mkLinkQuery info) [x,id] cursor //Create the link
		| isJust mbErr = (Just (GSQLDatabaseError (fromJust mbErr)), cursor)
		= linkRecord info xs id cursor

// --- Helper functions for GSQLUpdate mode --- //

updateOrInsertRecord :: [GSQLFieldInfo] [GSQLToken] [GSQLToken] *cur -> (Maybe GSQLError, GSQLToken, *cur) | SQLCursor cur
updateOrInsertRecord info values overrides cursor
	# (mbErr,upd,id,cursor)	= updateRecord info values overrides cursor
	| isJust mbErr		= (mbErr,id,cursor)
	| upd				= (Nothing,id,cursor)
	| otherwise			= insertRecord info values overrides cursor

updateRecord :: [GSQLFieldInfo] [GSQLToken] [GSQLToken] *cur -> (Maybe GSQLError, Bool, GSQLToken, *cur) | SQLCursor cur
updateRecord info values overrides cursor 
	# info						= filter store info 
	# stmt						= mkUpdateQuery info
	# values					= appOverrides info values overrides
	# idval						= hd values
	# (mbErr,cursor)			= sql_execute stmt (map fromValue (values ++ [hd values]) ) cursor
	| isJust mbErr				= (Just (GSQLDatabaseError (fromJust mbErr)),False, idval, cursor)
	# (mbErr,updated,cursor)	= sql_numRows cursor
	| isJust mbErr				= (Just (GSQLDatabaseError (fromJust mbErr)),False, idval, cursor)
	| updated == 0				= (Nothing, False, idval, cursor)
								= (Nothing, True, idval, cursor)

//Checks the link records and adds new links where necessary
relinkRecords :: [GSQLFieldInfo] [GSQLToken] GSQLToken !*cur -> (Maybe GSQLError, *cur) | SQLCursor cur
relinkRecords [] tokens id cursor = (Nothing, cursor)
relinkRecords [i:is] tokens id cursor
	# (values,tokens)	= takeTokens i.val_list tokens
	| isIndirect i
		# (mbErr, cursor) = relinkRecord i (map fromValue (init values)) (fromValue id) cursor
		| isJust mbErr	= (mbErr, cursor)
		# (mbErr, cursor) = cleanlinkRecord i (map fromValue (init values)) (fromValue id) cursor
		| isJust mbErr	= (mbErr, cursor)
		= relinkRecords is tokens id cursor
	| otherwise
		= relinkRecords is tokens id cursor
where
	relinkRecord info [] id cursor = (Nothing, cursor)
	relinkRecord info [x:xs] id cursor
		# (mbErr, cursor)		= sql_execute (mkCheckLinkQuery info) [x,id] cursor //Check if the link already exists
		| isJust mbErr			= (Just (GSQLDatabaseError (fromJust mbErr)), cursor)
		# (mbErr, num, cursor)	= sql_numRows cursor
		| isJust mbErr			= (Just (GSQLDatabaseError (fromJust mbErr)), cursor)
		| num > 0		
			= relinkRecord info xs id cursor
		| otherwise
			# (mbErr, cursor)	= sql_execute (mkLinkQuery info) [x,id] cursor //Create the link
			| isJust mbErr			= (Just (GSQLDatabaseError (fromJust mbErr)), cursor)
			= relinkRecord info xs id cursor

	cleanlinkRecord info values id cursor
		# (mbErr, cursor)		= sql_execute (mkCleanLinkQuery info values) [id:values] cursor
		| isJust mbErr			= (Just (GSQLDatabaseError (fromJust mbErr)), cursor)
								= (Nothing, cursor)

unlinkDirectRecords :: [GSQLFieldInfo] [GSQLToken] !*cur -> (Maybe GSQLError, *cur) | SQLCursor cur
unlinkDirectRecords [] tokens cursor = (Nothing, cursor)
unlinkDirectRecords [i:is] tokens cursor
	# (orig,tokens)	= takeTokens i.val_list tokens
	# (new,tokens)	= takeTokens i.val_list tokens
	| isDirect i && i.val_id
		= case (map fromValue (init (removedTokens orig new))) of
			[]			= unlinkDirectRecords is tokens cursor
			values
				# (mbErr, cursor)	= sql_execute (mkUnlinkDirectQuery i values) values cursor
				| isJust mbErr		= (Just (GSQLDatabaseError (fromJust mbErr)), cursor)
				= unlinkDirectRecords is tokens cursor
	| otherwise
		= unlinkDirectRecords is tokens cursor


// --- Helper functions for GSQLDelete mode --- //

deleteRecord :: [GSQLFieldInfo] GSQLToken !*cur -> (Maybe GSQLError, *cur ) | SQLCursor cur
deleteRecord info id cursor
	# (mbErr,cursor)	= sql_execute (mkDeleteQuery (hd info)) [fromValue id] cursor
	| isJust mbErr		= (Just (GSQLDatabaseError (fromJust mbErr)),cursor)
	= (Nothing, cursor)

unlinkRecords :: [GSQLFieldInfo] GSQLToken !*cur -> (Maybe GSQLError, *cur) | SQLCursor cur
unlinkRecords [] id cursor = (Nothing, cursor)
unlinkRecords [i:is] id cursor
	| isIndirect i
		# (mbErr,cursor)	= sql_execute (mkUnlinkQuery i) [fromValue id] cursor
		| isJust mbErr	= (Just (GSQLDatabaseError (fromJust mbErr)), cursor)
		= unlinkRecords is id cursor
	| otherwise
		= unlinkRecords is id cursor

// --- Predicates about fields --- //

//Checks if a record field's value should be stored now, or if it's 
//values are stored in a different database record.
store :: GSQLFieldInfo -> Bool
store info =: { fld_table, fld_select, rec_table} = (isJust fld_select) && (fld_table == rec_table)

//Checks if a field which is used for an incoming pointer/relation is direct, or uses a separate link table.
isDirect :: GSQLFieldInfo -> Bool
isDirect info =: {fld_table, val_fields}
	| isEmpty val_fields					= True
	| getTable val_fields == fld_table		= True
											= False
//Checks if a field is an indirect link
isIndirect :: GSQLFieldInfo -> Bool
isIndirect info =: { fld_select, fld_match} = (isJust fld_select) && (isJust fld_match)

//Checks if a field is an outgoing pointer
isOutPointer :: GSQLFieldInfo -> Bool
isOutPointer info =: {fld_table, fld_select, rec_table, val_fields} = (fld_table == rec_table) && (isJust fld_select) && (not (isEmpty val_fields))

// --- SQL generation functions --- //

mkSelectQuery :: !GSQLFieldInfo -> String
mkSelectQuery info
	# select = if (isNothing info.fld_select) (getKey info.val_fields) (fromJust info.fld_select)
	# match  = if (isNothing info.fld_match) info.rec_key (fromJust info.fld_match) 
	= "SELECT " +++ select +++ " FROM " +++ info.fld_table +++ " WHERE " +++ match +++ " = ?"

mkInsertQuery :: ![GSQLFieldInfo] -> String
mkInsertQuery info
	# table				= (hd info).fld_table
	# (fields,values)	= unzip [(fromJust i.fld_select,"?")  \\ i <- info]
	= "INSERT INTO " +++ table +++ " (" +++ (text_join "," fields) +++  ") VALUES (" +++ (text_join "," values) +++  ")"

mkUpdateQuery :: ![GSQLFieldInfo] -> String
mkUpdateQuery infos
	= "UPDATE " +++ table +++ " SET " +++ (text_join "," fields) +++ "WHERE " +++ match +++ " = ?"
	where
		table	= (hd infos).fld_table
		match	= fromJust (hd infos).fld_select
		fields	= [(fromJust info.fld_select) +++ " = ? " \\ info <- infos]

mkDeleteQuery :: !GSQLFieldInfo -> String
mkDeleteQuery info = "DELETE FROM " +++ info.fld_table +++ " WHERE " +++ (fromJust info.fld_select) +++ " = ?"

mkLinkQuery :: !GSQLFieldInfo -> String
mkLinkQuery info
	= "INSERT INTO " +++ info.fld_table +++ " (" +++ (fromJust info.fld_select) +++ "," +++ (fromJust info.fld_match) +++ ") VALUES (?,?)"

mkUnlinkQuery :: !GSQLFieldInfo -> String
mkUnlinkQuery info
	= "DELETE FROM " +++ info.fld_table +++ " WHERE " +++ (fromJust info.fld_match) +++ " = ?"

mkCheckLinkQuery :: !GSQLFieldInfo -> String
mkCheckLinkQuery info
	= "SELECT 0 FROM " +++ info.fld_table +++ " WHERE ( " +++ (fromJust info.fld_select) +++ " = ? ) AND ( " +++ (fromJust info.fld_match) +++ " = ? )"

mkCleanLinkQuery :: !GSQLFieldInfo ![SQLValue] -> String
mkCleanLinkQuery info []
	= "DELETE FROM " +++ info.fld_table +++ " WHERE " +++ (fromJust info.fld_match) +++ " = ?"
mkCleanLinkQuery info values
	= "DELETE FROM " +++ info.fld_table +++ " WHERE " +++ (fromJust info.fld_match) +++ " = ? AND NOT ( " +++ (fromJust info.fld_select) +++ " IN ( " +++ fields +++ " ))"
where
	fields = text_join "," ["?" \\ v <- values]

mkUnlinkDirectQuery :: !GSQLFieldInfo ![SQLValue] -> SQLStatement
mkUnlinkDirectQuery info values
	= "UPDATE " +++ info.fld_table +++ " SET " +++ (fromJust info.fld_match) +++ " = NULL WHERE " +++ (getKey info.val_fields) +++ " IN (" +++ fields +++ ")"
where
	fields = text_join "," ["?" \\ v <- values]

// --- Functions which manipulate token lists  --- //

//Merges the result token lists of two passes
mergeTokens :: [GSQLFieldInfo] [GSQLToken] [GSQLToken] -> [GSQLToken]
mergeTokens [] pass1 pass2 = []
mergeTokens [i:is] pass1 pass2 
	| store i	= x ++ (mergeTokens is xs pass2) with (x,xs) = (takeTokens i.val_list pass1)
	| otherwise	= x ++ (mergeTokens is pass1 xs) with (x,xs) = (takeTokens i.val_list pass2)

//Zips two token lists on a per field basis
zipTokens :: [GSQLFieldInfo] [GSQLToken] [GSQLToken] -> [GSQLToken]
zipTokens [] tokx toky = []
zipTokens [i:is] tokx toky = x ++ y ++ (zipTokens is xs ys)
where
	(x,xs) = takeTokens i.val_list tokx
	(y,ys) = takeTokens i.val_list toky

//Takes all the tokens for one field from the list, the Bool indicates if we need to take a list or a single value
takeTokens :: Bool [GSQLToken] -> ([GSQLToken],[GSQLToken])
takeTokens False [] = ([],[])
takeTokens False tokens = ([hd tokens],tl tokens)
takeTokens True [] = ([],[])
takeTokens True [GSQLTerminator:xs] = ([GSQLTerminator],xs)
takeTokens True [x:xs] = ([x:y],ys) where (y,ys) = takeTokens True xs

//Finds the tokens from the first list that are not present in the second
removedTokens :: [GSQLToken] [GSQLToken] -> [GSQLToken]
removedTokens orig new
	= [x \\ x <- init orig | not (isMember x (init new))] ++ [GSQLTerminator]

// --- Functions to piggyback overrides at the front of the tokenlist --- //

//Create the override tokens
mkOverrides :: [GSQLFieldInfo] GSQLToken -> [GSQLToken]
mkOverrides [] id = []
mkOverrides [i:is] id
	| isNothing i.fld_select	= [GSQLOverride (i.fld_table +++ "_" +++ (fromJust i.fld_match)) (fromValue id): mkOverrides is id]
								= mkOverrides is id

//Remove the overrides from the front of the token list
takeOverrides :: [GSQLToken] -> ([GSQLToken],[GSQLToken])
takeOverrides tokens = (takeWhile isOverride tokens, dropWhile isOverride tokens)
where
	isOverride (GSQLOverride f o)	= True
	isOverride _					= False

//Apply the overrides on a list of tokens
appOverrides :: [GSQLFieldInfo] [GSQLToken] [GSQLToken] -> [GSQLToken]
appOverrides info values overrides = [appOverride i v overrides \\ (i,v) <- (zip (info,values))]
where
	appOverride i v [] = v
	appOverride i v [x:xs]
		| (isNothing i.fld_select)	= v
		| otherwise = case x of
			(GSQLOverride f o)	= if ( f == (i.fld_table +++ "_" +++ (fromJust i.fld_select))) (GSQLValue o) v
			_					= v

// --- General helper functions --- //

instance == GSQLToken
where
	(==) (GSQLValue x)			(GSQLValue y)			= x == y
	(==) (GSQLTerminator)		(GSQLTerminator)		= True
	(==) (GSQLOverride xs xv)	(GSQLOverride ys yv)	= xs == ys && xv == yv
	(==) _						_						= False

//Tests if a given record is an ID (or an object)
isID :: String -> Bool
isID s = (s % ((size s) - 2, size s)) == "ID"

//Tests if a token is a value
isValue :: GSQLToken -> Bool
isValue (GSQLValue x)	= True
isValue _				= False

//Tests if a token is a value and is zero
isZeroValue :: GSQLToken -> Bool
isZeroValue (GSQLValue (SQLVInteger x))	= x == 0
isZeroValue _							= False

//Tests if a token is null value
isNullValue :: GSQLToken -> Bool
isNullValue (GSQLValue SQLVNull)		= True
isNullValue _							= False

fromValue :: GSQLToken -> SQLValue
fromValue (GSQLValue x) = x
fromValue _				= SQLVNull

getValues :: [GSQLToken] -> [SQLValue]
getValues tokens = map fromValue (takeWhile isValue tokens)

getKey :: [GSQLFieldInfo] -> String
getKey fields				= fromJust (hd fields).fld_select

getTable :: [GSQLFieldInfo] -> String
getTable fields			= (hd fields).fld_table

fetchIDToken :: *cur -> (Maybe SQLError, GSQLToken, *cur) | SQLCursor cur
fetchIDToken cursor
	# (err, val, cursor)	= sql_insertId cursor
	= (err, GSQLValue (SQLVInteger val), cursor)


//--- debug functions --//
instance toString GSQLToken
where
	toString (GSQLValue v)			= "GSQLValue " +++ (toString v)
	toString (GSQLTerminator)		= "GSQLTerminator"
	toString (GSQLOverride f v)		= "GSQLOverride " +++ f +++ " " +++ (toString v)

tokenString :: [GSQLToken] -> String
tokenString [] = ""
tokenString [x] = toString x
tokenString [x:xs] = toString x +++ ", " +++ tokenString xs
