
Module Nat Import Set;

(* This module defines the natural numbers by means of an inductive
   definition. Also a limited number of definitions and proofs are
   give. More can be found in Nat_more.l and other Nat_*.l files.
*)

Inductive [nat : SET]
Constructors [zeroN : nat] [succ : {n:nat} nat]
Double;

(* The construct the set of natural numbers, we choose Leibniz
   equality as the equivalence relation.

   We also could use an inductive equality, but the LEGO system is
   better suited for Leibniz equality because the Qrepl tactic needs a
   polymorhic relation.
*)

[     Nat : Set
          = QSet nat
];

[     ZeroN  : el Nat  = zeroN      ]
[     OneN   : el Nat  = succ ZeroN ]
[     TwoN   : el Nat  = succ OneN  ]
[     ThreeN : el Nat  = succ TwoN  ]
[     FourN  : el Nat  = succ ThreeN]
[     FiveN  : el Nat  = succ FourN ]
[     SixN   : el Nat  = succ FiveN ]
;

[     Succ : Fun Nat Nat
          = QFun|?|Nat succ
];

(* --------------------------------------------------------------------------------
   Define a few induction schemes and recursors.
*)

[    nat_rec [T|TYPE] = nat_elim nat\T
             : T -> ({n:nat}{ih:T}T) -> nat -> T
]
[    nat_iter [T|TYPE] = [x:T][f:T->T] nat_rec x (nat\f)
             : T -> ({ih:T}T) -> nat -> T
]
[    nat_ind [P:nat->Prop] = nat_elim P
             : (P zeroN) -> ({n:nat}{ih:P n} P (succ n)) -> {n:nat} P n 
];

Goal {phi:nat->TYPE} (phi ZeroN) -> (phi OneN) ->
     ({n:nat}(phi n.succ) -> (phi n.succ.succ)) -> {n:nat} phi n;
  intros ____;
  Refine nat_elim ? H;
  Refine nat_elim [n:nat] (phi n) -> phi n.succ;
    intros; Refine H1;
  intros __; Refine H2;
Save nat_elim2;

Goal {phi:nat->TYPE} (phi ZeroN) -> (phi OneN) -> (phi TwoN) ->
     ({n:nat}(phi n.succ.succ) -> (phi n.succ.succ.succ)) -> {n:nat} phi n;
  intros _____;
  Refine nat_elim2 ? H H1;
  Refine nat_elim [n:nat] (phi n.succ) -> phi n.succ.succ;
    intros; Refine H2;
  intros __; Refine H3;
Save nat_elim3;

Goal {phi:nat->TYPE} (phi ZeroN) -> (phi OneN) -> (phi TwoN) -> (phi ThreeN) ->
     ({n:nat}(phi n.succ.succ.succ) -> (phi n.succ.succ.succ.succ)) -> {n:nat} phi n;
  intros ______;
  Refine nat_elim3 ? H H1 H2;
  Refine nat_elim [n:nat] (phi n.succ.succ) -> phi n.succ.succ.succ;
    intros; Refine H3;
  intros __; Refine H4;
Save nat_elim4;

(* --------------------------------------------------------------------------------
   Define the predecessor.
*)

[     predN : Nat.el -> Nat.el
          = nat_rec ZeroN ([x,_:nat]x)
]
[     PredN : Fun Nat Nat
          = QFun predN
];

Goal {x:el Nat} Eq (PredN.ap (Succ.ap x)) x;
  Refine Eq_refl;
Save PredN_ok;

(* --------------------------------------------------------------------------------
   Prove some basic properties about Succ.
*)

Goal Succ_inj : Injection Succ;
  Intros ___;
  Refine Eq_trans (PredN.ap (Succ.ap a));
    Refine PredN_ok;
  Refine Eq_trans (PredN.ap (Succ.ap a'));
    Refine exten; Refine H;
  Refine PredN_ok;
Save;

Goal Succ_not_zero : {n:Nat.el} ~(Eq (Succ.ap n) ZeroN);
  Intros __;
  Refine H (nat_elim (nat\Prop) absurd (nat\Prop\trueProp));
  Refine trueprf;
Save;

Goal {x:el Nat} ~(Eq x (Succ.ap x));
  Refine nat_ind [x:el Nat] ~(Eq x (Succ.ap x));
  Intros _; Refine Succ_not_zero ? H.Eq_sym;
  Intros ___; Refine ih; Refine Succ_inj H;
Save Succ_lemma1;

Goal {x:el Nat} ~(Eq x ZeroN) -> Eq (Succ.ap (PredN.ap x)) x;
  Refine nat_elim [x:el Nat] ~(Eq x ZeroN) -> Eq (Succ.ap (PredN.ap x)) x;
  intros; Refine H; Refine Eq_refl;
  intros; Refine Succ_inj; Refine PredN_ok;
Save PredN_lemma1;

(* --------------------------------------------------------------------------------
   Show the natural numbers are a discrete set.
*)

Goal Nat_discr : Discrete Nat;
  Refine nat_double_elim [x,y:el Nat] (Eq x y) \/ ~(Eq x y);
  orIL; Refine Eq_refl;
  intros; orIR; notI; Refine Succ_not_zero ? H.Eq_sym;
  intros; orIR; notI; Refine Succ_not_zero ? H;
  intros; orE n_ih n';
    intros; orIL; Refine exten Succ H;
    intros; orIR; notI; notE H; Refine Succ_inj H1;
Save;

(* --------------------------------------------------------------------------------
   Define the addition and some convenient lemmas.
*)

[     plusN : bop Nat.el
          = [m:el Nat] nat_iter m Succ.ap
]
[     PlusN : Fun2 Nat Nat Nat
          = QFun2 plusN
];

Goal lZeroN_ident : lIdentity PlusN ZeroN;
  Refine nat_ind [n:el Nat] Eq (PlusN.ap2 ZeroN n) n;
  Refine Eq_refl;
  intros; Equiv Eq (Succ.ap (PlusN.ap2 ZeroN n)) (Succ.ap n);
    Qrepl ih; Refine Eq_refl;
Save;

Goal rZeroN_ident : rIdentity PlusN ZeroN;
  Refine Eq_refl;
Save;

Goal ZeroN_ident : Identity PlusN ZeroN;
  Refine pair lZeroN_ident rZeroN_ident;
Save;

Freeze rZeroN_ident lZeroN_ident ZeroN_ident;

Goal {x,y:Nat.el} Eq (PlusN.ap2 x (Succ.ap y)) (Succ.ap (PlusN.ap2 x y));
  intros; Refine Eq_refl;
Save rPlusSuccN;

Goal {x,y:Nat.el} Eq (PlusN.ap2 (Succ.ap x) y) (Succ.ap (PlusN.ap2 x y));
  intros _;
  Refine nat_ind [y:el Nat] Eq (PlusN.ap2 (Succ.ap x) y) (Succ.ap (PlusN.ap2 x y));
  Refine Eq_refl;
  intros; Refine exten Succ ih;
Save lPlusSuccN;

Goal PlusN_assoc : Associative PlusN;
  Intros m n l;
  Refine nat_ind [l:nat] Q (plusN m (plusN n l)) (plusN (plusN m n) l);
  Refine Q_refl;
  intros; Refine exten Succ ih;
Save;

Goal PlusN_commut : Commutative PlusN;
  Refine nat_double_elim [m,n:nat] Q (plusN m n) (plusN n m);
  Refine Q_refl;
  intros; Refine exten Succ n_ih;
  intros; Refine exten Succ (n_ih ZeroN);
  intros;
    Refine exten Succ;
    Qrepl (lPlusSuccN n n'); Qrepl (lPlusSuccN n' n);
    Refine exten Succ (n_ih n');
Save;

Freeze PlusN_assoc PlusN_commut;

(* --------------------------------------------------------------------------------
   Define the multiplication and some convenient lemmas.
*)

[     timesN : bop Nat.el
          = [m:el Nat] nat_iter ZeroN (PlusN.ap2 m)
]
[     TimesN : Fun2 Nat Nat Nat
          = QFun2 timesN
];

Goal rTimesZeroN : {x:el Nat} Eq (TimesN.ap2 x ZeroN) ZeroN;
  intros; Refine Eq_refl;
Save;

Goal lTimesZeroN : {x:el Nat} Eq (TimesN.ap2 ZeroN x) ZeroN;
  Refine nat_ind [x:el Nat] Eq (TimesN.ap2 ZeroN x) ZeroN;
  Refine Eq_refl;
  intros;
  Refine Eq_trans (TimesN.ap2 ZeroN n); 
    Refine lZeroN_ident;
  Refine ih;
Save;

Goal {x,y:el Nat} Eq (TimesN.ap2 x (Succ.ap y)) (PlusN.ap2 x (TimesN.ap2 x y));
  intros;
  Refine Eq_refl;
Save rTimesSuccN;

Goal {x,y:el Nat} Eq (TimesN.ap2 (Succ.ap x) y) (PlusN.ap2 (TimesN.ap2 x y) y);
  intros _;
  Refine nat_ind [y:el Nat]
                  Eq (TimesN.ap2 (Succ.ap x) y) (PlusN.ap2 (TimesN.ap2 x y) y);
  Refine Eq_refl;
  intros y ih;
  Refine Eq_trans (PlusN.ap2 (succ x) (PlusN.ap2 (TimesN.ap2 x y) y));
    Refine exten2 ? ?.Eq_refl ih;
  Refine Eq_trans (Succ.ap (PlusN.ap2 x (PlusN.ap2 (TimesN.ap2 x y) y)));
    Refine lPlusSuccN;
  Refine exten; Refine PlusN_assoc;
Save lTimesSuccN;

Goal rOneN_ident : rIdentity TimesN OneN;
  Refine Q_refl;
Save;

Goal lOneN_ident : lIdentity TimesN OneN;
  Refine nat_ind [n:nat] Q (timesN OneN n) n;
  Refine Q_refl;
  intros; Equiv Q (PlusN.ap2 OneN (TimesN.ap2 OneN n)) (Succ.ap n);
    Qrepl ih; Qrepl (PlusN_commut OneN n); Refine Q_refl;
Save;

Goal OneN_ident : Identity TimesN OneN;
  Refine pair lOneN_ident rOneN_ident;
Save;

Goal lTimesPlusN_distrib : lDistributive PlusN TimesN;
  Intros a b;
  Refine nat_ind [c:nat] Q (timesN a (plusN b c)) (plusN (timesN a b) (timesN a c));
  Refine Q_refl;
  intros c _;
  Refine Eq_trans (PlusN.ap2 a (plusN (timesN a b) (timesN a c)));
    Refine exten2 PlusN; Refine Q_refl; Immed;
  Refine Eq_trans (PlusN.ap2 (plusN a (timesN a b)) (timesN a c)); Refine PlusN_assoc;
  Qrepl (PlusN_commut a (TimesN.ap2 a b));
  Refine Eq_trans (PlusN.ap2 (timesN a b) (plusN a (timesN a c)));
  Refine Q_sym; Refine PlusN_assoc;
  Refine Q_refl;
Save;

Goal rTimesPlusN_distrib : rDistributive PlusN TimesN;
  Refine nat_ind [a:nat] {b,c:nat}
                 Q (timesN (plusN b c) a) (plusN (timesN b a) (timesN c a));
  intros; Refine Q_refl;
  intros a ___;
  Refine Eq_trans (PlusN.ap2 b (plusN (timesN b a) (plusN c (timesN c a))));
    Refine +1 PlusN_assoc;
  Refine Eq_trans (PlusN.ap2 (plusN b c) (plusN (timesN b a) (timesN c a)));
    Refine exten2 PlusN; Refine Q_refl; Immed;
  Refine Eq_trans (PlusN.ap2 b (plusN c (plusN (timesN b a) (timesN c a))));
    Refine Q_sym; Refine PlusN_assoc;
  Refine exten2 PlusN; Refine Q_refl;
  Refine Eq_trans (PlusN.ap2 (plusN c (timesN b a)) (timesN c a)); Refine PlusN_assoc;
  Refine Eq_trans (PlusN.ap2 (plusN (timesN b a) c) (timesN c a));
  Refine exten2 PlusN; Refine PlusN_commut; Refine Q_refl;
  Refine Q_sym; Refine PlusN_assoc;
Save;

Goal TimesPlusN_distrib : Distributive PlusN TimesN;
  Refine pair lTimesPlusN_distrib rTimesPlusN_distrib;
Save;

Goal Associative TimesN;
  Intros x y;
  Refine nat_ind [z:el Nat]
    Eq (TimesN.ap2 x (TimesN.ap2 y z)) (TimesN.ap2 (TimesN.ap2 x y) z);
  Refine Eq_refl;
  intros z _;
  Refine Eq_trans (PlusN.ap2 (TimesN.ap2 x y) (TimesN.ap2 x (TimesN.ap2 y z)));
  Refine lTimesPlusN_distrib;
  Refine exten2; Refine Eq_refl; Immed;
Save TimesN_assoc;

Goal Commutative TimesN;
  Refine nat_ind [x:el Nat]{y:el Nat} Eq (TimesN.ap2 x y) (TimesN.ap2 y x);
  intros;
    Refine Eq_trans ZeroN;
      Refine lTimesZeroN;
    Refine Eq_sym; Refine rTimesZeroN;
  intros x ih y;
    Refine Eq_trans (PlusN.ap2 (TimesN.ap2 x y) y);
      Refine lTimesSuccN;
    Refine Eq_trans (PlusN.ap2 (TimesN.ap2 y x) y);
      Refine exten2 ? ?.ih ?.Eq_refl;
    Refine Eq_trans (PlusN.ap2 y (TimesN.ap2 y x));
      Refine PlusN_commut;
    Refine rTimesSuccN;
Save TimesN_commut;

(* --------------------------------------------------------------------------------
   Miscellaneous.
*)

[     EmptySET_nat : EmptySET -> nat
          = EmptySET_elim (EmptySET\nat)
];
