// ************************************************************
//	Clean Linear Algebra Subroutines - CLAS
//	Version 0.7 - June 23, 1999 - Thorsten Zoerner
// 	Catholic University of Nijmegen - zoerner@cs.kun.nl
// ************************************************************

implementation module slicingExamples

import slicing

sliceGauss :: *Matrix -> *Matrix
sliceGauss aa = gauss_k 0 a
where
	(n, a) = usize aa
	gauss_k k a 
		| k==n = a
		#! a = sliceCE (Col k2n k) (/) (Ele k k) a
		#! a = sliceSCR (Sli k2n k2n) (-) (Col k2n k) (*) (Row k k2n) a
		= gauss_k (inc k) a
	where
		k2n = [inc k .. dec n]

sliceCholesky :: *Matrix -> *Matrix
sliceCholesky aa = chol_k 0 a
where
	(n, a) = usize aa
	chol_k :: Int *Matrix -> *Matrix
	chol_k k a =: { [k, k] = akk}
		#! a = { a & [k, k] = sqrt akk}
		| k==dec n = a
		#! a = sliceCE (Col k2n k) (/) (Ele k k) a
		= chol_k (inc k) (chol_j (inc k) a)
	where
		k2n = [inc k .. dec n]
		chol_j :: Int *Matrix -> *Matrix
		chol_j j a
			| j==n = a
			#! a = sliceCCE (Col j2n j) (-) (Col j2n k) (*) (Ele j k) a
			= chol_j (inc j) a
		where
			j2n = [j .. dec n]
		
solveLSP :: *Matrix .Vector -> .Vector
solveLSP a b
	#! c = (transpose a) ** b
// ---------------------------------------
//	Choose one of the following two lines:
	# r = qr_house a
//	# r = qr_givens a
// ---------------------------------------
	# y = forwardSubst (transpose r) c
	= backwardSubst r y 

qr_house :: *Matrix -> *Matrix
qr_house a
	#! m = size a.[0]
	= qr_j 0 m a
where
	qr_j j m a 
		| j==m = a
		= qr_j (inc j) m (house_j j a) 

house :: .Vector -> .(Vector, Real)
house x
	#! v = { xx \\ xx <-: x}
	#! v = { v & [0] = 1.0}
	#! sigma = dot x x - (x0 * x0)
	| sigma==0.0 = (v, 0.0)
	#! mu = sqrt (x0 * x0 + sigma)
  	#! v0 = calc_v0 x0 sigma mu
  	#! v02 = v0 * v0
	= ( { v & [0] = v0} /. v0, 2.0 * v02 / (sigma + v02))
where
	n = length x
	x0 = x.[0]
	calc_v0 :: Real Real Real -> Real
	calc_v0 x0 sigma mu
		| x0 <= 0.0 = x0 - mu
		= ~ sigma / (x0 + mu)

house_j :: Int *Matrix -> .Matrix
house_j j a
	#! n = size a
	#! m = size a.[0]
	#! j2n = [j .. dec n]
	#! j2m = [j .. dec m]
	#! x = cutC (Col j2n j) a
	#! (v, beta) = house x
	#! w = beta .* (transposeSlice (Sli j2n j2m, a) ** v)
	= sliceSVV (Sli j2n j2m) (-) v (*) w a 

qr_givens :: *Matrix -> *Matrix
qr_givens a 
	#! n = size a
	#! m = size a.[0] 
	= qrg_j 0 n m a
where
	qrg_j :: Int Int Int *Matrix -> *Matrix
	qrg_j j n m a
		| j==m = a
		= qrg_j (inc j) n m (qrg_i (dec n) m a)
	where
		qrg_i :: Int Int *Matrix -> *Matrix
		qrg_i i m a
			| i==j = a
			= qrg_i (dec i) m (givens_ij i j m a)
			
givens_ij :: Int Int Int *Matrix -> *Matrix
givens_ij i j m a =: { [j, j] = ajj, [i, j] = aij} 
	#! (c, s) = givens ajj aij
	#! qij = { { c, ~s}, { s, c}}
	= sliceMipS qij (Sli [j, i] [j .. dec m]) a  
	
givens_ij2 :: Int Int Int *Matrix -> *Matrix
givens_ij2 i j m a =: { [di, j] = adij, [i, j] = aij} 
	#! (c, s) = givens adij aij
	#! qij = { { c, ~s}, { s, c}}
	= sliceMipS qij (Sli [di, i] [j .. dec m]) a  
where
	di = dec i
