module MacTetris;

/*
	A Concurrent Clean variant of Tetris.
	
	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 1800K.
	The linker needs an additional 700K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 700K of free memory is needed
	outside the Clean 0.8 application.
*/

import	StdClass;
import  deltaEventIO, deltaIOSystem, deltaMenu, deltaTimer, deltaDialog;
import  deltaWindow, deltaPicture, StdFile, deltaSystem;
import  StdInt, StdString, StdBool, StdArray;
import  Tetris, Help;


     
	FileMenuID		:== 1;
		NewGameID 	:== 11;
		PauseID		:== 12;
		HighID		:== 13;
		QuitID		:== 14;

	HighDlogID		:== 1;
		HighTitleID	:== 11;
		HighOKID		:== 12;
	OverDlogID		:== 2;
		OverOKID		:== 21;

	WindowID 	:== 1;

	TimerID		:== 1;

	MaxDelay 		:== TicksPerSecond / 4;
	MinDelay 		:== TicksPerSecond / 30;
	DelayLevel		:==	100;
	ZeroCount		:==	0;
	NrOfHiScores	:== 8;

	PicDomain	:== ((0,0),(SizeX,SizeY));
	SizeX				:== 200;
	SizeY				:== 310;
	WindowCorner	:== (0,0);

	InitialState seed	:== (InitialBoard,StartPos seed,seed);

	HiScoresFile	:== "tetrishi";
	HelpFile			:== "MacTetrisHelp";


    

::	* UFILE	:== * File;
:: *MyIO			:==  IOState MyState;
::	* MyState	:==	* (State, Delay, Counter, HiFiles);
:: *IO_div_State		:== (MyState, MyIO);
::	Delay			:== Int;
::	Counter		:== Int;
::	*HiFiles		:== (Files,HiScores);
::	HiScores		:== [(String, Int)];


    

Start	:: * World -> * World;
Start world =  CloseEvents events` (closefiles (WriteHiScores hifile highs`) world``);
	where {
	(s,c,p,highs`)  =: state;
	(state ,events`)=: PlayTetris highs events;
	(hifile,highs)  =: ReadHiScores HiScoresFile files;
	(events,world``)=: OpenEvents world`;
	(files ,world` )=: openfiles world;
	};

InitialiseRandomSeed :: MyState MyIO -> IO_div_State;
InitialiseRandomSeed (_,cd,ct,(files,hs)) io
	=	((InitialState seed,cd,ct,(files,hs)), io`);
	where {
		((hours,minutes,seconds), io`)
			= GetCurrentTime io;
		seed
			= seconds + minutes * 60 + hours * 3600 + 1
	};

PlayTetris	:: HiFiles EVENTS -> (MyState,EVENTS);
PlayTetris (files,highs) events
	= 	StartIO [about, window, menus, timer] (StartMyState (files`,highs) 0) [InitialiseRandomSeed] events;

		where {
		about	=: DialogSystem [about_dialog];
		(about_dialog,files`)=: MakeAboutDialog "MacTetris" HelpFile files Help;

		window=: WindowSystem [FixedWindow WindowID WindowCorner "Tetris"
									 	PicDomain Redraw [GoAway Ignore, Keyboard Unable UnpackKey]];

		timer	 =: TimerSystem	[Timer TimerID Unable MaxDelay UnpackTimer];

		menus	 =: MenuSystem [tetris];
		tetris =: PullDownMenu FileMenuID "MacTetris" Able [
						MenuItem NewGameID	"New Game" 	(Key 'R') Able   NewGame,
						MenuItem PauseID		"Halt" 		(Key '.') Unable Halt,
						MenuSeparator,
						MenuItem HighID "High Scores" (Key 'H') Able ShowHiScores,
						MenuSeparator,
						MenuItem QuitID "Quit" (Key 'Q') Able Quit];
		};

StartMyState :: HiFiles RandomSeed -> MyState;
StartMyState highs seed =  (InitialState seed, MaxDelay, ZeroCount, highs);

Ignore	:: MyState MyIO -> IO_div_State;
Ignore mystate io =  (mystate, io);

Help	:: MyState MyIO -> IO_div_State;
Help (st,cd,ct,(files,hs)) io =  ((st,cd,ct,(files`,hs)), io`);
		where {
		(files`,io`)=: ShowHelp HelpFile files io;
		};


/*	The MenuFunctions:
*/

Quit	:: MyState MyIO -> IO_div_State;
Quit mystate io =  (mystate, QuitIO io);

NewGame	:: MyState MyIO -> IO_div_State;
NewGame (st=:(_,_,seed),cd,ct,highs) io
	= 	(StartMyState highs seed, io`);
		where {
		io`=: ChangeIOState [ ActivateWindow WindowID,
									EnableMenuItems [PauseID],
									DisableMenuItems [NewGameID, HighID, QuitID],
									ChangeMenuItemFunctions [(PauseID, Halt)],
									ChangeMenuItemTitles [(PauseID, "Halt")],
									EnableKeyboard WindowID,
									EnableTimer TimerID,
									DrawInWindow WindowID pic] io;
		pic=: [EraseRectangle PicDomain :
		         Concat DrawBorder (DrawFirstScore_and_Level MaxDelay MaxDelay ZeroCount)];
		};

Halt	:: MyState MyIO -> IO_div_State;
Halt mystate io
	= 	(mystate, io`);
		where {
		io`=: ChangeIOState [ DisableTimer TimerID,
									EnableMenuItems  [NewGameID, QuitID],
									DisableKeyboard WindowID,
									ChangeMenuItemFunctions [(PauseID, Continue)],
									ChangeMenuItemTitles [(PauseID, "Continue")]] io;
		};

Continue	:: MyState MyIO -> IO_div_State;
Continue mystate io
	= 	(mystate, io`);
		where {
		io`=: ChangeIOState [ ActivateWindow   WindowID,
									DisableMenuItems [NewGameID, QuitID],
									EnableKeyboard   WindowID,
									ChangeMenuItemFunctions [(PauseID, Halt)],
									ChangeMenuItemTitles [(PauseID, "Halt")],
									EnableTimer TimerID] io;
		};


/*	The TimerFunction:
*/

UnpackTimer	:: TimerState MyState MyIO -> IO_div_State;
UnpackTimer timepast mystate io =  DoBlockMove Tick mystate io;


/*	The KeyboardFunction:
*/

UnpackKey :: KeyboardState MyState MyIO -> IO_div_State;
UnpackKey (key,KeyUp   ,mod_new) mystate io =  (mystate, io);
UnpackKey ('j',keystate,mod_new) mystate io =  DoBlockMove ToLeft  mystate io;
UnpackKey ('k',keystate,mod_new) mystate io =  DoBlockMove Turn    mystate io;
UnpackKey ('l',keystate,mod_new) mystate io =  DoBlockMove ToRight mystate io;
UnpackKey (' ',KeyDown,mod_new)  mystate io =  DoBlockMove Drop    mystate io;
UnpackKey any					  mystate io =  (mystate, io);


DoBlockMove	:: Mode MyState MyIO -> IO_div_State;
DoBlockMove mode  mystate=:(state, delay, count, highs) io
	| gameover = 	GameOver (state`, MaxDelay, newcount, highs) io4;
	= 	((state`, newdelay, newcount, highs), io3) ;
		where {
		newcount=: Count landed count;	
		io4=: ChangeIOState [ DisableMenuItems [PauseID],
									EnableMenuItems  [NewGameID, HighID, QuitID],
									DisableKeyboard  WindowID,
									DisableTimer     TimerID] io3;
		io3=: DrawInWindow WindowID pic io2; 
		io2=: DrawInWindow WindowID (DrawScore_and_Level MaxDelay newdelay newcount) io1;
		(newdelay,io1)=: SetDelay landed delay newcount io;
		(state`, gameover, landed, pic)=: DrawBlockMove mode state;
		};
		
Count	:: Bool Int	-> Int;
Count True count	= 	count + 4;
Count false count	= 	count;
	
SetDelay	:: Bool Int Int MyIO -> (Int, MyIO);
SetDelay landed curdelay count io
	| landed && ( count mod DelayLevel  == 0)	= 	(newdelay, newio);
	= 	(curdelay, io);
		where {
		newdelay=: CalcNewDelay curdelay;
		newio	  =: SetTimerInterval TimerID newdelay io;
		};

CalcNewDelay	:: Int -> Int;
CalcNewDelay curdelay	| MinDelay > delay	= 	MinDelay;
									= 	delay;
										where {
										delay=: curdelay -  TicksPerSecond / 30 ;
										};


/*	The WindowUpdateFunction:
*/

Redraw	:: UpdateArea MyState -> (MyState, [DrawFunction]);
Redraw ua state=:((board, pos, seed), curdelay, count, highs)
   | count > 0	= 	(state, Concat (DrawPos pos) pic`);
   =  (state, [font : pic`]);	 //	don't show start position of game not yet started
   	where {
   	pic`=: Concat DrawBorder (Concat (DrawBoard board) (
   			   DrawFirstScore_and_Level MaxDelay curdelay count));
   	font=: SetFont myfont;
   	(b,myfont)=: SelectFont "New York" ["Bold"] 14;
   	};


/*	Show the high scores:
*/

ShowHiScores	:: MyState MyIO -> IO_div_State;
ShowHiScores state=:(st,cd,p,(files,[])) io
	=  (state, Alert "No high scores available." io);
ShowHiScores state=:(st,cd,p,(files,highs)) io
	= 	OpenModalDialog dialog state io;
		where {
		dialog=: CommandDialog HighDlogID "High Scores" [] HighOKID [title : scores];
		title=:  StaticText HighTitleID Center "MacTetris Hall of Fame:";
		scores=: MakeScores 1 highs;
		};

MakeScores	:: Int HiScores -> [DialogItem MyState MyIO];
MakeScores id [] =  [DialogButton HighOKID Center "OK" Able HighOK];
MakeScores id [(name, hi) : scores]
	= 	[dt, st : MakeScores (inc id) scores];
		where {
		dt=: DynamicText id (YOffset (dec id) (Pixel 2)) (MM 65.0) ( toString id  +++  ". " +++ name );
		st=: StaticText (id + 20) (XOffset id (MM 0.0)) (toString hi);
		};

HighOK	:: DialogInfo MyState MyIO -> IO_div_State;
HighOK dialog state io =  (state, CloseActiveDialog io);

/*	Game Over, check for a new high score:
*/

GameOver	:: MyState MyIO -> IO_div_State;
GameOver state=:(st,cd,points,(files,hiscores)) io
	| ItsAHighScore points hiscores = 	OpenModalDialog dialog state io;
	= 	(state, Alert "Game Over, no high score." io);
		where {
		dialog=: CommandDialog OverDlogID "Game Over"
				      [ItemSpace (MM 6.0) (MM 6.0)] OverOKID [st1,st2,et,ok];
		st1	=: StaticText 1 Left  "Game Over with a new high score!";
		st2	=: StaticText 2 Left  "Your name:";
		et		=: EditText   3 (RightTo 2) (MM 45.0) 1 "";
		ok		=: DialogButton OverOKID Center "OK" Able OverOK;
		};

OverOK	:: DialogInfo MyState MyIO -> IO_div_State;
OverOK dialog state=:(st,cd,points,(files,hiscores)) io
	| name == "" = 	(state, CloseActiveDialog io);
	= 	((st,cd,points,(files,hiscores`)), CloseActiveDialog io);
		where {
		hiscores`=: Take NrOfHiScores (AddScore (String13 name) points hiscores);
		name     =: GetEditText 3 dialog;
		};

ItsAHighScore	:: Int HiScores -> Bool;
ItsAHighScore points scores
	| points == 0 = 	False;
	|  Length_new scores  < NrOfHiScores = 	True;
	= 	IsItReallyAHighScore points scores;

IsItReallyAHighScore	:: Int HiScores -> Bool;
IsItReallyAHighScore points [] =  False;
IsItReallyAHighScore points [(name,score):rest]
	| points > score = 	True;
	= 	IsItReallyAHighScore points rest;

AddScore	:: String Int HiScores -> HiScores;
AddScore name points [] =  [(name,points)];
AddScore name points scores=:[score=:(n,pts):rest]
	| pts > points = 	[score : AddScore name points rest];
	= 	[(name,points) : scores];


/*	General Alert dialog.
*/

Alert	:: String MyIO -> MyIO;
Alert mes io =  io`;
		where {
		(id,io`)=: OpenNotice notice io;
		notice	=: Notice [mes] (NoticeButton 1 "OK") [];
		};


/*	Highscores file handling:
*/

ReadHiScores	:: String Files -> (UFILE, HiFiles);
ReadHiScores fname files
	| exists = 	(file`, (files`,highs));
	= 	(create, (files``,[]));
		where {
		(exists, file, files`)    =: fopen fpath FReadData files;
		(highs, file`)            =: ReadHighs exists file;
		(success, create, files``)=: fopen fpath FWriteData files`;
		fpath                     =: HomePath fname;
		};

ReadHighs	:: Bool UFILE -> (HiScores, UFILE);
ReadHighs False file =  ([],file);
ReadHighs b file
	| eof = 	([], file1);
	= 	([(name, hi) : rest], file5);
		where {
		(eof,file1)  =: fend file;
		(name,file2) =: freads file1 13;
		(b1,hi,file3)=: freadi file2;
		(b2,nl,file4)=: freadc file3;
		(rest, file5)=: ReadHighs (b1 && b2) file4;
		};


WriteHiScores	:: UFILE HiFiles -> Files;
WriteHiScores file (files,highs) =  files`;
		where {
		(closed, files`)=: fclose (WriteHighs success file` highs) files;
		(success, file`)=: freopen file FWriteData;
		};

WriteHighs	:: Bool UFILE HiScores -> UFILE;
WriteHighs False file highs =  file;
WriteHighs b file [] =  file;
WriteHighs b file [(name,hi):scores]
	= 	WriteHighs b file` scores;
		where {
		file`=: fwritec '\n' (fwritei hi (fwrites name` file));
		name`=: ThirteenCharString name;
		};


/*	Auxiliary functions:
*/

Length_new	:: [x]   -> Int;
Length_new []    =  0;
Length_new [x:y] =  inc (Length_new y);

Concat	:: [x] [x] -> [x];
Concat [] ys	=  ys;
Concat [x:xs] ys =  [x: Concat xs ys];

Take	:: Int [x] -> [x];
Take 0 list  =  [];
Take n [x:y] =  [x : Take (dec n) y];
Take n list  =  list;

String13	:: String -> String;
String13 string |  size string  > 13 =  string % (0, 12);
						 =  string;

ThirteenCharString	:: String -> String;
ThirteenCharString string =  (string +++ "             ") % (0, 12); // RWS
