module Hanoi;

/*
	The Towers of Hanoi (graphical version).

	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 1500K.
	The linker needs an additional 500K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 600K of free memory is needed
	outside the Clean 0.8 application.
*/

import StdEnv, deltaEventIO, deltaMenu;
import deltaDialog, deltaTimer, deltaWindow, deltaPicture;

    
::	Tower			:== [Int];
::	Moves			:== [Int];
::	* Towers	:== * (Moves,Tower,Tower,Tower);
::	* IO		:==  IOState Towers;
::	* TowIO	:== (Towers,IO);

     
	FileID		:== 1;
	RunID			:== 11;
	HaltID		:== 12;
	SpeedID		:== 13;
		VerSID 	:== 131;
		SlowID	:== 132;
		NormID	:== 133;
		FastID	:== 134;
		VerFID	:== 135;
	QuitID		:== 14;
	TimerID		:== 1;
	WindowID		:== 1;
	Corner		:== (0,0);
	ScrBar		:== ScrollBar (Thumb 0) (Scroll 10);
	PicDom		:== ((50,0),(480,180));
	Speed1		:== TicksPerSecond / 2;
	Speed2		:== TicksPerSecond / 3;
	Speed3		:== TicksPerSecond / 6;
	Speed4		:== TicksPerSecond / 12;
	Speed5		:== 0;
	MinDisks		:== 2;
	MaxDisks		:== 12;
	XOffs			:==  inc MaxDisks  * 10;
	
    

//	The I/O system

Start	:: * World -> * World;
Start world =  CloseEvents events` world`;
	where {
	(towers,events`)=: StartIO [menus,windw,timer] ([],[],[],[]) [] events;
	(events,world` )=: OpenEvents world;
	menus=: MenuSystem [file];
	file =: PullDownMenu FileID "File" Able
	        [SubMenuItem RunID "Run" Able (RunItems MinDisks MaxDisks),
	         MenuItem HaltID "Halt" (Key '.') Unable Halt,
	         SubMenuItem SpeedID "Speed" Able [speed],
	         MenuItem QuitID "Quit" (Key 'Q') Able Quit
	        ];
	speed=: MenuRadioItems NormID
	        [MenuRadioItem VerSID "Very Slow" (Key 'A') Able (SetSpeed Speed1),
		      MenuRadioItem SlowID "Slow"      (Key 'S') Able (SetSpeed Speed2),
		      MenuRadioItem NormID "Normal"    (Key 'D') Able (SetSpeed Speed3),
		 	   MenuRadioItem FastID "Fast"      (Key 'F') Able (SetSpeed Speed4),
		      MenuRadioItem VerFID "Very Fast" (Key 'G') Able (SetSpeed Speed5)
		 	  ];
	windw=: WindowSystem [wndw];
	wndw =: FixedWindow WindowID Corner "Hanoi" PicDom Update_new [GoAway Ignore];
	timer=: TimerSystem [Timer TimerID Unable Speed3 StepHanoi];
	};

RunItems	:: Int Int -> [MenuElement Towers IO];
RunItems i last
	| i > last =  [];
	=  [item : RunItems (inc i) last];
	where {
	item   =: MenuItem (i + 1000) itemstr NoKey Able (Run i);
	itemstr=: If (i > MinDisks) istr (istr +++ " Disks");
	istr   =: toString i;
	};

//	The function for the Run commands

Run	:: Int Towers IO -> TowIO;
Run nrdisks tow io
	=  DrawInActiveWindowFrame Update_new towers run;
	where {
	towers=: (Hanoi nrdisks 1 2 3,MakeTower nrdisks [],[],[]);
	run   =: EnableTimer TimerID menus;
	menus =: DisableMenuItems [RunID, QuitID] (EnableMenuItems [HaltID] io);
	};

MakeTower	:: Int Tower -> Tower;
MakeTower 0 tower =  tower;
MakeTower n tower =  MakeTower (dec n) [n:tower];

//	The function for the Halt command

Halt	:: Towers IO -> TowIO;
Halt tow io =  (([],[],[],[]), DisableTimer TimerID menus);
	where {
	menus=: EnableMenuItems [RunID, QuitID] (DisableMenuItems [HaltID] io);
	};

//	The Quit command function

Quit	:: Towers IO -> TowIO;
Quit tow io =  (tow, QuitIO io);

//	Set the speed of a (possibly running) Hanoi simulation

SetSpeed	:: Int Towers IO -> TowIO;
SetSpeed speed tow io =  (tow, SetTimerInterval TimerID speed io);

//	The timer function: take a move from the lazy list of moves and show
//	it in the window

StepHanoi	:: TimerState Towers IO -> TowIO;
StepHanoi ts tow=:([],t1,t2,t3) io =  (tow,DisableTimer TimerID menus);
	where {
	menus=: EnableMenuItems [RunID, QuitID] (DisableMenuItems [HaltID] io);
	};
StepHanoi ts ([from_new,to:rest],t1,t2,t3) io =  ((rest,t1`,t2`,t3`),io`);
	where {
	(t1`,t2`,t3`,io`)=: ChangeTowers from_new to t1 t2 t3 io;
	};

ChangeTowers	:: Int Int Tower Tower Tower IO -> (Tower,Tower,Tower,IO);
ChangeTowers 1 2 [f:r] t2 t3 io =  (r,[f:t2],t3,draw);
	where {
	draw=: DrawInActiveWindow [DrawMove 1 2 f (Length_new r) (Length_new t2)] io;
	};
ChangeTowers 1 3 [f:r] t2 t3 io =  (r,t2,[f:t3],draw);
	where {
	draw=: DrawInActiveWindow [DrawMove 1 3 f (Length_new r) (Length_new t3)] io;
	};
ChangeTowers 2 1 t1 [f:r] t3 io =  ([f:t1],r,t3,draw);
	where {
	draw=: DrawInActiveWindow [DrawMove 2 1 f (Length_new r) (Length_new t1)] io;
	};
ChangeTowers 2 3 t1 [f:r] t3 io =  (t1,r,[f:t3],draw);
	where {
	draw=: DrawInActiveWindow [DrawMove 2 3 f (Length_new r) (Length_new t3)] io;
	};
ChangeTowers 3 1 t1 t2 [f:r] io =  ([f:t1],t2,r,draw);
	where {
	draw=: DrawInActiveWindow [DrawMove 3 1 f (Length_new r) (Length_new t1)] io;
	};
ChangeTowers 3 2 t1 t2 [f:r] io =  (t1,[f:t2],r,draw);
	where {
	draw=: DrawInActiveWindow [DrawMove 3 2 f (Length_new r) (Length_new t2)] io;
	};

DrawMove	:: Int Int Int Int Int Picture -> Picture;
DrawMove from_new to disk lenfr lento pic
	=  DrawRectangle ((tx - w,ty),(tx + w,ty + 10)) erase;
	where {
	erase=: EraseRectangle ((fx - w,fy),(fx + w,fy + 10)) pic;
	tx   =: to *   XOffs;  ty=: 10 +  10 * (MaxDisks - lento) ;
	fx   =: from_new * XOffs;  fy=: 10 +  10 * (MaxDisks - lenfr) ;
	w    =: disk * 5;
	};

//	The GoAway function: ignore requests to close the window

Ignore	:: Towers IO -> TowIO;
Ignore tow io =  (tow, io);

//	The update function: erase the window and redraw the towers

Update_new	:: UpdateArea Towers -> (Towers, [DrawFunction]);
Update_new area tow=:(ms,t1,t2,t3) =  (tow,[erase,draw1,draw2,draw3]);
	where {
	erase=: EraseRectangle PicDom;
	draw1=: DrawTower 1 (MaxDisks -  Length_new t1 ) t1;
	draw2=: DrawTower 2 (MaxDisks -  Length_new t2 ) t2;
	draw3=: DrawTower 3 (MaxDisks -  Length_new t3 ) t3;
	};

DrawTower	:: Int Int Tower Picture -> Picture;
DrawTower nr i [f:r] pic =  DrawTower nr (inc i) r disk;
	where {
	disk=: DrawRectangle ((x - w,y),(x + w,y + 10)) pic;
	x   =: nr * XOffs;  w=: f * 5;  y=: 20 +  10 * i ;
	};
DrawTower 1 i [] pic
	=  DrawString "from" (MovePenTo (XOffs - 14,35 +  10 * MaxDisks ) pic);
DrawTower 2 i [] pic
	=  DrawString "to" (MovePenTo ( 2 * XOffs  - 6,35 +  10 * MaxDisks ) pic);
DrawTower 3 i [] pic
	=  DrawString "via" (MovePenTo ( 3 * XOffs  - 9,35 +  10 * MaxDisks ) pic);

//	The function that calculates the lazy list of disk moves

Hanoi	:: Int Int Int Int -> Moves;
Hanoi 0 t1 t2 t3 =  [];
Hanoi n from_new to via
	=  Concat (Hanoi m from_new via to) [from_new,to : Hanoi m via to from_new]; where { m=: dec n; };

//	Miscellaneous functions

Concat	:: [x]   [x]  -> [x];
Concat []    list =  list;
Concat [f:r] list =  [f : Concat r list];

Length_new	:: [x] -> Int;
Length_new [] =  0;
Length_new [f:r] =  inc (Length_new r);

If	:: Bool  x x -> x;
If True  x y =  x;
If false x y =  y;
