module LifeGame;

/*
	A more sophisticated version of the game of Life_
	Several generations can be shown in different windows
	and a help command is included.
	
	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 1800K.
	The linker needs an additional 500K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 800K of free memory is needed
	outside the Clean 0.8 application.
*/

import	StdClass;
import StdInt, StdString, StdFile;
import deltaEventIO, deltaMenu, deltaTimer, deltaWindow, deltaPicture, deltaSystem;
import Life, Help;

	
    

::	* Lives		:== (Size, RunningWindows, [Life], Files);
::	Life			:== (Generation, Size, WindowId);
::	RunningWindows	:== (WindowId, [WindowId]);
::	Size			:== Int;
::	*IO				:==  IOState Lives;


     

	FileMenuID :== 1;
		NewID   :== 11;
		CloseID :== 12;
		HelpID  :== 13;
		QuitID  :== 14;
	OptionsMenuID :== 2;
		EraseID :== 21;
		CellSizeID :== 22;
			Size1ID   :== 2201;		Size1	:== 1;
			Size2ID   :== 2202;		Size2	:== 2;
			Size4ID   :== 2204;		Size4	:== 4;
			Size8ID   :== 2208;		Size8	:== 8;
			Size16ID  :== 2216;		Size16	:== 16;
	CommandsMenuID :== 3;
		PlayID :== 31;
		HaltID :== 32;
		StepID :== 33;
	LivesMenuID :== 4;
		LivesGroupID :== 41;
			
	BaseLifeMenuItemID :== LivesGroupID * 10;
			
	SizeItem id t size	:== MenuRadioItem id t NoKey Able (ChangeSize size);
	Item id t c s f		:== MenuItem id t (Key c) s f;
	SelectItem id		:== CheckMenuItem (SelectID id) ("Life " +++  toString id )
										 NoKey Able Mark (SelectLife id);
	SelectID id			:== BaseLifeMenuItemID + id;
	SizeID size			:==  CellSizeID * 100  + size;
	
	LifeWindow id :== ScrollWindow id (id * 10,id * 10) ("Life" +++  toString id )
								  (ScrollBar (Thumb 0) (Scroll 8)) (ScrollBar (Thumb 0) (Scroll 8))
								  ((-10000,-10000), (10000,10000)) (70,70) (300,200)
								  (UpdateWindow id) [Activate (MakeWindowActive id),
								  Deactivate DeActive, GoAway (CloseThisWindow id),
								  Mouse Able (Track id), Cursor CrossCursor];
	BaseWindowID   :== 1;
	
	TimerID        :== 1;
	  NullInterval :== 0;
	
	HelpFile :== "LifeHelp";


    

/*	The Start rule initiates the program's interaction. The I/O system consists of a Window-,
	Menu- and TimerDevice. 
	The programstate contains the current cellsize (Size) and all open windows (Lives). Each 
	window shows one game of life (Life). The initial cellsize is 8 by 8 pixels, and the 
	application opens with no windows.
*/

Start	:: * World -> * World;
Start world =  CloseEvents events` (Lives_CloseFiles state world``);
		where {
		(state,events`)=: StartIO [about, windows, menus, null] (StartState files`) [] events;
		(files,world``)=: openfiles world`;
		(events,world`)=: OpenEvents world;
		
		about	=: DialogSystem [about_dialog];
		(about_dialog, files`)=: MakeAboutDialog "LifeGame" HelpFile files Help;
		
		windows	=: WindowSystem [LifeWindow 1];
		
		null	=: TimerSystem [Timer TimerID Unable NullInterval CalcGenerations];
		
		menus	=: MenuSystem [file, options, commands, lives];
		file	=: PullDownMenu FileMenuID "File" Able [
						Item NewID	 "New"	 'N' Able New,
						Item CloseID "Close" 'W' Able CloseTheActiveWindow,
						MenuSeparator,
						Item HelpID  "Help"  '/' Able Help,
						MenuSeparator,
						Item QuitID	 "Quit"	 'Q' Able Quit];
		options =: PullDownMenu OptionsMenuID "Options" Able [
						Item EraseID "Erase All Cells" 'E' Unable Erase,
						SubMenuItem CellSizeID "Cell Size" Able [
							MenuRadioItems Size8ID [
								SizeItem Size1ID  " 1 * 1"	Size1,
								SizeItem Size2ID  " 2 * 2"	Size2,
								SizeItem Size4ID  " 4 * 4"	Size4,
								SizeItem Size8ID  " 8 * 8"	Size8,
								SizeItem Size16ID "16 * 16"	Size16]]];
		commands=: PullDownMenu CommandsMenuID "Commands" Able [
						Item PlayID "Play" 'P' Able Play,
						Item HaltID "Halt" 'H' Unable Halt,
						Item StepID "Step" 'S' Able Step];
		lives	=: PullDownMenu LivesMenuID "Lives" Able [
						MenuItemGroup LivesGroupID [SelectItem 1]];
		};


StartState	:: Files -> Lives;
StartState files =  (Size8, (0, []), [([], Size8, 1)], files);

/*	The MenuFunctions of the PullDownMenu "File":
*/

New	:: Lives IO -> (Lives, IO);
New lives io
	= 	(lives1, io1);
		where {
		io1=: ChangeIOState [OpenWindows [LifeWindow window_id],
							InsertMenuItems LivesGroupID 0 [SelectItem window_id]] io;
		(window_id, lives1)=: CreateNewLife lives;
		};
	
CloseTheActiveWindow	:: Lives IO -> (Lives, IO);
CloseTheActiveWindow lives io
	| mine2		= 	(lives1, io2);
	= 	SetNoneActiveWindow lives1 io2;
		where {
		lives1=: OneLifeLess active_id1 (Lives_RemoveRunningWindow active_id1 lives);
		(mine2, active_id2, io2)=: GetActiveWindow (RemoveMenuItems [SelectID active_id1] (
												   CloseWindows [active_id1] io1));
		(mine1, active_id1, io1)=: GetActiveWindow io;
		};
	
SetNoneActiveWindow	:: Lives IO -> (Lives, IO);
SetNoneActiveWindow lives io
	= 	(lives, ChangeIOState [DisableMenuItems [CloseID, EraseID, PlayID, HaltID, StepID],
							   DisableTimer TimerID] io);

Help	:: Lives IO -> (Lives, IO);
Help (size, running, lifes, files) io =  ((size, running, lifes, files`), io`);
		where {
		(files`,io`)=: ShowHelp HelpFile files io;
		};

Quit	:: Lives IO -> (Lives, IO);
Quit lives io =  (lives, QuitIO io);


/*	The MenuFunctions of the PullDownMenu "Options":
*/

Erase	:: Lives IO -> (Lives, IO);
Erase lives io
	= 	(lives1, io2);
		where {
		lives1=: Lives_SetLifeGeneration lives window_id [];
		io2	  =: DrawInActiveWindow [EraseRectangle ((0,0), (1000,1000))] io1;
		(mine, window_id, io1)=: GetActiveWindow io;
		};

ChangeSize	:: Int Lives IO -> (Lives, IO);
ChangeSize new_size lives io
	= 	ChangeScrollBar window_id (ChangeScrolls new_size new_size) lives2 io2;
		where {
		lives2=: Lives_SetSize (Lives_SetLifeSize lives1 window_id new_size) new_size;
		io2	  =: DrawInActiveWindow [DrawNewSizes generation new_size] io1;
		(generation, size)	  =: Life_Attributes life;
		(found, life, lives1) =: Lives_Life window_id lives;
		(mine, window_id, io1)=: GetActiveWindow io;
		};


/*	The MenuFunctions of the PullDownMenu "Commands":
*/

Play	:: Lives IO -> (Lives, IO);
Play lives io
	= 	(lives2, io2);
		where {
		lives2=: Lives_AddRunningWindow window_id lives1;
		io2	  =: ChangeIOState [EnableMenuItems	[HaltID],
							   DisableMenuItems	[PlayID, StepID, EraseID],
							   DisableMouse		window_id,
							   EnableTimer TimerID] io1;
		(found, life, lives1) =: Lives_Life window_id lives;
		(mine, window_id, io1)=: GetActiveWindow io;
		};

Halt	:: Lives IO -> (Lives, IO);
Halt lives io
		| life_goes_on= 	(lives3, io2);
	= 	(lives3, DisableTimer TimerID io2);
		where {
		(life_goes_on, lives3)=: LifeGoesOn lives2;
		lives2=: Lives_RemoveRunningWindow window_id lives1;
		io2	  =: ChangeIOState [DisableMenuItems	[HaltID],
							   EnableMenuItems	[EraseID, PlayID, StepID],
							   EnableMouse		window_id] io1;
		(found, life, lives1) =: Lives_Life window_id lives;
		(mine, window_id, io1)=: GetActiveWindow io;
		};

Step	:: Lives IO -> (Lives, IO);
Step lives io
	= 	(lives2, io2);
		where {
		lives2=: Lives_SetLifeGeneration lives1 window_id next;
		io2	  =: DrawInActiveWindow [DrawNewGeneration new died size] io1;
		(next, new, died)	  =: LifeGame generation;
		(generation, size)	  =: Life_Attributes life;
		(found, life, lives1) =: Lives_Life window_id lives;
		(mine, window_id, io1)=: GetActiveWindow io;
		};


/*	The MenuFunctions of the PullDownMenu "Lives":
*/

SelectLife	:: WindowId Lives IO -> (Lives, IO);
SelectLife window_id lives io
		| window_id == now_window_id= 	(lives, io1);
	= 	(lives, ActivateWindow window_id io1);
		where {
		(mine, now_window_id, io1)=: GetActiveWindow io;
		};
	
	
/*	The TimerDevice calculates and draws the next generation of one running game of Life_ If there
	are N games of life running, it will take N timer-events to compute for each game of life the
	next generation.
*/

CalcGenerations	:: TimerState Lives IO -> (Lives, IO);
CalcGenerations t lives io
		| window_id == 0= 	(lives2, io);
	= 	(lives3, io1);
		where {
		lives3=: Lives_SetLifeGeneration lives2 window_id next;
		io1	  =: DrawInWindow window_id [DrawNewGeneration new died size] io;
		(next, new, died)	 =: LifeGame generation;
		(generation, size)	 =: Life_Attributes life;
		(found, life, lives2)=: Lives_Life window_id lives1;
		(window_id, lives1)	 =: Lives_NextRunningWindow lives;
		};


/*	Each Window displays one game of Life_ LifeWindows accept only mouse-events to place and
	remove life-cells, defined by the function Track. Updating the window redraws the complete
	current generation, without regard of the UpdateArea. The window is closed by pressing it's
	close-region (function CloseThisWindow). Also the Life administration is made garbage.
*/

Track	:: WindowId MouseState Lives IO -> (Lives, IO);
Track my_window (pos, ButtonUp, mod_new) lives io 
	= 	(lives, io);
Track my_window ((x,y), buttondown, CommandOnly) lives io
	= 	(lives2, io1);
		where {
		(found, life, lives1)=: Lives_Life my_window lives;
		lives2		 =: Lives_SetLifeGeneration lives1 my_window (RemoveCell (xc, yc) generation);
		io1			 =: DrawInActiveWindow [EraseRectangle cellpos] io;
		cellpos		 =: ((nx, ny),(nx + size, ny + size));
		xc			 =: nx / size;
		yc			 =: ny / size;
		nx			 =: x -  x mod size ;
		ny			 =: y -  y mod size ;
		(generation, size)=: Life_Attributes life;
		};
Track my_window ((x,y), buttondown, mod_new) lives io
	= 	(lives2, io1);
		where {
		(found, life, lives1)=: Lives_Life my_window lives;
		lives2		 =: Lives_SetLifeGeneration lives1 my_window (InsertCell (xc, yc) generation);
		io1			 =: DrawInActiveWindow [FillRectangle cellpos] io;
		cellpos		 =: ((nx, ny),(nx + size, ny + size));
		xc			 =: nx / size;
		yc			 =: ny / size;
		nx			 =: x -  x mod size ;
		ny			 =: y -  y mod size ;
		(generation, size)=: Life_Attributes life;
		};

UpdateWindow	:: WindowId UpdateArea Lives -> (Lives, [DrawFunction]);
UpdateWindow my_window upd_area lives
	= 	(lives1, [Draw generation size]);
		where {
		(generation, size)	 =: Life_Attributes life;
		(found, life, lives1)=: Lives_Life my_window lives;
		};

MakeWindowActive	:: WindowId Lives IO -> (Lives, IO);
MakeWindowActive active_id lives io
								| in_play_mode= 	(lives3, set_play_items);
		| Empty gen= 	(lives3, DisableMenuItems [EraseID] set_sow_items);
	= 	(lives3, set_sow_items);
		where {
		lives3			=: Lives_SetSize lives2 size;
		set_sow_items	=: ChangeIOState [EnableMenuItems [PlayID, StepID, EraseID],
										  DisableMenuItems [HaltID]] mark_life_items;
		set_play_items	=: ChangeIOState [DisableMenuItems [PlayID, StepID, EraseID],
										  EnableMenuItems [CloseID, HaltID]] mark_life_items;
		mark_life_items	=: ChangeIOState [UnmarkMenuItems all_life_items,
										  MarkMenuItems [SelectID active_id],
										  UnmarkMenuItems all_size_items,
										  MarkMenuItems [SizeID size]] io;
		all_life_items	=: WindowIdsToMenuItemIds all_life_windows;
		all_size_items	=: [Size1ID, Size2ID, Size4ID, Size8ID, Size16ID];
		(all_life_windows, lives2)=: Lives_WindowIds lives1;
		(gen, size)				  =: Life_Attributes life;
		(found, life,  lives1)	  =: Lives_Life active_id lives0;
		(in_play_mode, lives0)    =: InPlayMode active_id lives;
		};

DeActive	:: Lives IO -> (Lives, IO);
DeActive lives io =  (lives,io);

WindowIdsToMenuItemIds	:: [WindowId] -> [MenuItemId];
WindowIdsToMenuItemIds [id : ids] =  [SelectID id : WindowIdsToMenuItemIds ids];
WindowIdsToMenuItemIds []		  =  [];

CloseThisWindow	:: WindowId Lives IO -> (Lives, IO);
CloseThisWindow window_id lives io
			| mine= 	MakeWindowActive active_id lives1 io1;
	= 	SetNoneActiveWindow lives1 io1;
		where {
		lives1				  =: OneLifeLess window_id (Lives_RemoveRunningWindow window_id lives);
		(mine, active_id, io1)=: GetActiveWindow (
									ChangeIOState [CloseWindows [window_id],
												   RemoveMenuItems [SelectID window_id]] io);
		};
	
	
/*	The drawing functions:
*/

DrawNewGeneration	:: Generation Generation Int Picture -> Picture;
DrawNewGeneration new died cellsize picture
	= 	Draw new cellsize (
			SetPenColour BlackColour (
			Draw died cellsize (
			SetPenColour WhiteColour picture)));
		 
DrawNewSizes	:: Generation Int Picture -> Picture;
DrawNewSizes generation cellsize picture
	= 	Draw generation cellsize (EraseRectangle ((0,0),(1000,1000)) picture);

Draw	:: Generation Int Picture -> Picture;
Draw [[(x,y) : z_x] : z_xs] cellsize picture
	= 	Draw [z_x : z_xs] cellsize (FillRectangle ((lbx,lby),(ubx,uby)) picture);
		where {
		lbx=: x * cellsize;
		lby=: y * cellsize;
		ubx=: lbx + cellsize;
		uby=: lby + cellsize;
		};
Draw [[] : z_xs] cell_size picture
	= 	Draw z_xs cell_size picture;
Draw empty cellsize picture
	= 	picture;


/*	Access-rules on Life :: (Generation, Size, WindowId)
*/

Life_Attributes	:: Life -> (Generation, Size);
Life_Attributes (gen, size, window_id) =  (gen, size);
	
Life_SetGeneration	:: Life Generation -> Life;
Life_SetGeneration (gen, size, window_id) gen` =  (gen`, size, window_id);

Life_SetSize	:: Life Size -> Life;
Life_SetSize (gen, size, window_id) size` =  (gen, size`, window_id);

Empty	:: [x] -> Bool;
Empty [] =  True;
Empty no =  False;
	

/*	Access-rules on Lives :: (Size, RunningWindows, [Life], FILES)
*/

Lives_CloseFiles	:: Lives * World -> * World;
Lives_CloseFiles (size, running, lifes, files) world =  closefiles files world;

Lives_Life	:: WindowId Lives -> (Bool, Life, Lives);
Lives_Life window_id lives=:(size, running, lifes, files)
	= 	(found, life, lives);
		where {
		(found, life)=: Lifes_Life window_id lifes;
		};
	
Lifes_Life	:: WindowId [Life] -> (Bool, Life);
Lifes_Life window_id [] =  (False, ([],0,0));
Lifes_Life window_id [life=:(gen, size, id) : lives]
		| window_id == id= 	(True, life);
	= 	Lifes_Life window_id lives;
	
Lives_SetSize	:: Lives Size -> Lives;
Lives_SetSize (size, running, lives, files) size` =  (size`, running, lives, files);
	
Lives_SetLifeGeneration	:: Lives WindowId Generation -> Lives;
Lives_SetLifeGeneration (size, running, lifes, files) window_id gen
	= 	(size, running, Lifes_SetGeneration lifes window_id gen, files);
	
Lifes_SetGeneration	:: [Life] WindowId Generation -> [Life];
Lifes_SetGeneration [life=:(gen, size, id) : lives] window_id gen`
		| window_id == id= 	[(gen`, size, id) : lives];
	= 	[life : Lifes_SetGeneration lives window_id gen`];
Lifes_SetGeneration [] window_id gen` =  [];

Lives_SetLifeSize	:: Lives WindowId Size -> Lives;
Lives_SetLifeSize (size, running, lifes, files) window_id size`
	= 	(size, running, Lifes_SetSize lifes window_id size`, files);
	
Lifes_SetSize	:: [Life] WindowId Size -> [Life];
Lifes_SetSize [life=:(gen, size, id) : lives] window_id size`
		| window_id == id= 	[(gen, size`, id) : lives];
	= 	[life : Lifes_SetSize lives window_id size`];
Lifes_SetSize [] window_id size` =  [];
	
Lives_WindowIds	:: Lives -> ([WindowId], Lives);
Lives_WindowIds lives=:(size, running, [], files)	 =  ([], lives);
Lives_WindowIds lives=:(size, running, lifes, files) =  (Lifes_WindowIds lifes, lives);

Lifes_WindowIds	:: [Life] -> [WindowId];
Lifes_WindowIds [(gen, size, id) : lives] =  [id : Lifes_WindowIds lives];
Lifes_WindowIds [] =  [];
		
InPlayMode	:: WindowId Lives -> (Bool, Lives);
InPlayMode active_id lives=:(size, (id, []),      lifes, files) =  (False, lives);
InPlayMode active_id lives=:(size, (id, running), lifes, files)
	=  (InPlayMode` active_id running, lives);
	
InPlayMode`	:: WindowId [WindowId] -> Bool;
InPlayMode` active_id [id : ids]
		| active_id == id= 	True;
	= 	InPlayMode` active_id ids;
InPlayMode` active_id [] =  False;

LifeGoesOn	:: Lives -> (Bool, Lives);
LifeGoesOn lives=:(size, (id, []), lifes, files) =  (False, lives);
LifeGoesOn lives =  (True, lives);

Lives_AddRunningWindow	:: WindowId Lives -> Lives;
Lives_AddRunningWindow window_id (size, (id, []), lifes, files)
	= 	(size, (id, [window_id]), lifes, files);
Lives_AddRunningWindow window_id (size, (id, running), lifes, files)
	= 	(size, (id, AddRunningWindow window_id running), lifes, files);
	
AddRunningWindow	:: WindowId [WindowId] -> [WindowId];
AddRunningWindow window_id [id : ids]
		| window_id <= id= 	[window_id, id : ids];
	= 	[id : AddRunningWindow window_id ids];
AddRunningWindow window_id [] =  [window_id];

Lives_RemoveRunningWindow	:: WindowId Lives -> Lives;
Lives_RemoveRunningWindow window_id (size, (id, []), lifes, files)
	=  (size, (id, []), lifes, files);
Lives_RemoveRunningWindow window_id (size, (id, running), lifes, files)
	= 	(size, (id, RemoveRunningWindow window_id running), lifes, files);
	
RemoveRunningWindow	:: WindowId [WindowId] -> [WindowId];
RemoveRunningWindow window_id the_ids=:[id : ids]
			| window_id == id= 	ids;
		| window_id < id= 	the_ids;
	= 	[id : RemoveRunningWindow window_id ids];
RemoveRunningWindow window_id [] =  [];

Lives_NextRunningWindow	:: Lives -> (WindowId, Lives);
Lives_NextRunningWindow lives=:(size, (id, []), lifes, files)
	= 	(0, lives);
Lives_NextRunningWindow (size, (id, the_ids=:[first_id : ids]), lifes, files)
	= 	(id`, (size, (id`, the_ids), lifes, files));
		where {
		id`=: NextRunningWindow first_id id the_ids;
		};

NextRunningWindow	:: WindowId WindowId [WindowId] -> WindowId;
NextRunningWindow first_id id [] =  first_id;
NextRunningWindow first_id id [id` : ids]
		| id < id`= 	id`;
	= 	NextRunningWindow first_id id ids;
	
CreateNewLife	:: Lives -> (WindowId, Lives);
CreateNewLife (size, running, lifes, files)
	= 	(new_window_id, (size, running, lifes`, files));
		where {
		(new_window_id, lifes`)=: CreateNewLife` size 1 lifes;
		};
	
CreateNewLife`	:: Size WindowId [Life] -> (WindowId, [Life]);
CreateNewLife` size` window_id all_lives=:[life=:(gen, size, id) : lives]
		| window_id == id= 	(window_id`, [life : new_lives]);
	= 	(window_id, [([], size`, window_id) : all_lives]);
		where {
		(window_id`, new_lives)=: CreateNewLife` size` (inc window_id) lives;
		};
CreateNewLife` size` window_id [] =  (window_id, [([], size`, window_id)]);

OneLifeLess	:: WindowId Lives -> Lives;
OneLifeLess window_id (size, running, lifes, files)
	=  (size, running, OneLifeLess` window_id lifes, files);
	
OneLifeLess`	:: WindowId [Life] -> [Life];
OneLifeLess` window_id [life=:(gen, size, id) : lives]
		| window_id <> id= 	[life : OneLifeLess` window_id lives];
	= 	lives;
OneLifeLess` window_id [] =  [];
