implementation module windowDevice


import deltaIOSystem, StdEnv, misc
import oskernel,ostypes,oswindow,osdebug
import ioState


from deltaWindow import CloseWindows
from menuDevice import OpenWindowMenu
from windowDef import IsScrollWindow
from commonDef import SelectStateEqual
from picture import StartDrawing, EndDrawing, StartUpdate, EndUpdate, NewPicture

		
::  DeltaControl :== ControlPtr -> ControlPtr


MaxScrollBar :== 32000


WGetWindowPtr:: Window -> WindowPtr
WGetWindowPtr (winptr, pic, hscroll, vscroll, cursor, offset)
		= winptr

		
WGetPicturePtr:: Window -> PicturePtr
WGetPicturePtr (winptr, pic, hscroll, vscroll, cursor, offset)
		= pic


WGetVScrollState:: Window -> ScrollState
WGetVScrollState (winptr, pic, hscroll, vscroll, cursor, offset)
		= vscroll

		
WGetHScrollState:: Window -> ScrollState
WGetHScrollState (winptr, pic, hscroll, vscroll, cursor, offset)
		= hscroll

		
WGetCursorPtr:: Window -> CursorPtr
WGetCursorPtr (winptr, pic, hscroll, vscroll, cursor, offset)
		= cursor

		
WGetOffset:: Window -> Offset
WGetOffset (winptr, pic, hscroll, vscroll, cursor, offset)
		= offset

		
Window_SetTitle:: !Window !WindowTitle -> Window
Window_SetTitle (w_ptr, pic, h_bar, v_bar, cursor, offset) title
		= let!
			wptr1 = OSWinSetTitle w_ptr title
		  in (wptr1, pic, h_bar, v_bar, cursor, offset)


/*  General access-rules:
*/

Window_SetThumbs:: !Window !Int !Int -> Window
Window_SetThumbs (wptr,pic,hscroll,vscroll,cursor,(oldx,oldy)) newx newy
		= let!
			windowptr` = ScrollWindowIfNeeded wptr hdiff` vdiff`
		  in (windowptr`,pic,hscroll`,vscroll`,cursor,(hpos`,vpos`))
		  where
		  hdiff = newx - oldx
		  vdiff = newy - oldy
		  (hdiff`, hpos`, hscroll`) = SetScrollPosition hdiff oldx hscroll
		  (vdiff`, vpos`, vscroll`) = SetScrollPosition vdiff oldy vscroll


Window_FrameSize:: !Window -> (!Int, !Int)
Window_FrameSize (winptr, pic, hscroll, vscroll, cursor, offset)
		= (width, height)
		  where
		  (width, height,_,_) = OSWinGetSizeAndPos winptr


Window_GetScrolls:: !Window -> (!Int, !Int)
Window_GetScrolls (w, p, hscroll, vscroll, c, o)
		= (hscroll.scroll_line, vscroll.scroll_line)


MakePageScroll:: Int Int -> Int
MakePageScroll winsize linesize
	| winsize > linesize
		= ((winsize / linesize) - 1) * linesize
		= linesize


Window_SetScrolls:: !Window !Int !Int -> Window
Window_SetScrolls (winptr, p, hscroll, vscroll, c, o) hline` vline`
		= let!
			(width, height,_,_) = OSWinGetSizeAndPos winptr
		  in (winptr, p, hscroll`, vscroll`, c, o)
		  where
		  hpage`   = MakePageScroll width hline`
		  vpage`   = MakePageScroll height vline`
		  hscroll` = {hscroll & scroll_line = hline`, scroll_page = hpage`}
		  vscroll` = {vscroll & scroll_line = vline`, scroll_page = vpage`}


Window_GetThumbs:: !Window -> (!Int, !Int)
Window_GetThumbs (w, p, h, v, c, offset)
		= offset


WindowFunctions:: DeviceFunctions s
WindowFunctions
		= (ShowWindow, OpenWindow, WindowIO, CloseWindow, HideWindow)


OpenWindow:: !.(DeviceSystem s (IOState s)) !(IOState s) -> IOState s
OpenWindow (WindowSystem w_defs) io_state
		= let! 
			(io_state`, windows) = Open_windows w_defs (io_state,[])
		  in IOStateSetDevice io_state` (WindowSystemState windows)


OpenNewWindows:: ![WindowDef s (IOState s)] !(!IOState s) -> (!IOState s, !WindowHandles s)
OpenNewWindows wdefs io
		= OpenNewWindows` wdefs (IOStateGetDevice io WindowDevice)
		  where
		  
		  OpenNewWindows`:: ![WindowDef s (IOState s)] !(!DeviceSystemState s, !IOState s) -> (!IOState s, !WindowHandles s)
		  OpenNewWindows` wdefs (WindowSystemState whandles, io)
				= Open_windows wdefs (io,whandles)


Open_windows:: ![WindowDef s (IOState s)] !(!IOState s, !WindowHandles s) -> (!IOState s, !WindowHandles s)
Open_windows [w_def : w_defs] io_wins
		= Open_windows w_defs (Open_windows` (ValidateWindow w_def) io_wins)
Open_windows w_defs           io_wins
		= io_wins

Open_windows`:: !(WindowDef s (IOState s)) !(!IOState s, !WindowHandles s) -> (! IOState s, ! WindowHandles s)
Open_windows` (ScrollWindow id pos title hbar vbar domain msize isize upd atts) (io, windows)
	| WindowIdNotUsed id windows
		= let!
			winptr             = OSWinCreate title pos isize True True
			(winptr`,io`)      = OpenWindowAndMenu winptr io
			(ix`, iy`,_,_)     = OSWinGetSizeAndPos winptr`
			(hscroll, vscroll, offset) = MakeScrolls hbar vbar domain (ix`, iy`) winptr`
			winptr``           = OSWinActivate winptr`
			cursor             = GetWindowCursor atts
			picture            = NewPicture winptr``
			w_def`             = ScrollWindow id pos title hbar vbar domain msize isize upd (AddDefaultGoAway id atts)
			window             = (winptr``, picture, hscroll, vscroll , cursor, offset)
		  in (io`, [(w_def`,window) : windows])
		= (io, windows)
Open_windows` (FixedWindow id pos title domain upd atts) (io, windows)
	| WindowIdNotUsed id windows
		= let!
			winptr        = OSWinCreate title pos (SizeOfDomain domain) False False
			winptr``      = OSWinActivate winptr`
			cursor        = GetWindowCursor atts
			picture       = NewPicture winptr``
			w_def`        = FixedWindow id pos title domain upd (AddDefaultGoAway id atts)
			window        = (winptr``, picture, DummyScroll, DummyScroll, cursor, offset)
			(winptr`,io`) = OpenWindowAndMenu winptr io
			(offset, _)   = domain
		  in (io`, [(w_def`,window) : windows])
		= (io, windows)


MakeScrolls:: !ScrollBarDef !ScrollBarDef !PictureDomain !Size !WindowPtr -> (!ScrollState, !ScrollState, (Int, Int))
MakeScrolls (ScrollBar (Thumb xpos) (Scroll hline)) (ScrollBar (Thumb ypos) (Scroll vline))
		  ((xmin, ymin), (xmax, ymax)) (xinit,yinit) windowptr
		= (hscroll, vscroll, (xpos`, ypos`))
		  where
		  (hscroll, xpos`) = SetScrollValues hscrollptr xmin xmax xpos xinit hline
		  (vscroll, ypos`) = SetScrollValues vscrollptr ymin ymax ypos yinit vline
		  hscrollptr       = OSWinGetHScroll windowptr
		  vscrollptr       = OSWinGetVScroll windowptr


SetScrollValues :: !ControlPtr !Int !Int !Int !Int !Int -> (!ScrollState, !Int)
SetScrollValues scrollptr min2 max2 pos init line
		= (scroll, pos`) 
		  where
		  scroll     = { scroll_ptr  = scrollptr`,
		                 scroll_line = line, 
		                 scroll_page = MakePageScroll init line,
		                 scroll_min  = min2,
		                 scroll_max  = max`,
		                 scroll_div  = div
		                }
		  max`        = max min2 (max2 - init)
		  pos`        = max min2 (min pos max`)
		  pos``       = Align_thumb pos` min2 max` line 
		  div         = getdivider min2 max`
		  spos        = (pos`` - min2) >> div
		  smax        = (max` - min2) >> div
		  sinit       = init >> div
		  stotal      = (max2 - min2) >> div
		  spos`       = if (div > 0 && pos`` >= max`) (spos + 1) spos
		  scrollptr`  = OSWinSetScrollValues scrollptr spos` smax sinit stotal

		  getdivider :: !Int !Int -> Int
		  getdivider min max
				= f max` 0
				  where
				  max` = max - min
				  f m n
				  	| m >= MaxScrollBar
				  		= f (m >> 1) (n + 1)
				  		= n 

		  
OpenWindowAndMenu:: !WindowPtr !(IOState s) -> (!WindowPtr, !IOState s)
OpenWindowAndMenu wptr io
		= (wptr, OpenWindowMenu io)


SizeOfDomain:: ((Int,Int),(Int,Int)) -> (Int,Int)
SizeOfDomain ((xmin, ymin),(xmax, ymax))
		= (xmax - xmin,ymax - ymin)


DummyScroll:: ScrollState
DummyScroll
		= {scroll_ptr  = 0,
		   scroll_line = 0, 
		   scroll_page = 0,
		   scroll_min  = 0,
		   scroll_max  = 0,
		   scroll_div  = 0
		  }


AddDefaultGoAway:: !WindowId ![WindowAttribute s (IOState s)] -> [WindowAttribute s (IOState s)]
AddDefaultGoAway id atts =: [GoAway func : rest]
		= atts
AddDefaultGoAway id [attr : rest]
		= [attr : AddDefaultGoAway id rest]
AddDefaultGoAway id atts
		= [GoAway (DefaultGoAway id)]
		  where
		  DefaultGoAway id s io
				= (s, CloseWindows [id] io)


WindowIdNotUsed:: !Int !(WindowHandles s) -> Bool
WindowIdNotUsed id [w =: (windef, window) : windows]
	| id == WindowDef_WindowId windef 
		= Evaluate_2 False (OSWinActivate (WGetWindowPtr window))
		= WindowIdNotUsed id windows
WindowIdNotUsed id windows
		= True


CloseWindow:: !(IOState s) -> IOState s
CloseWindow io_state
		= Evaluate_2 (IOStateRemoveDevice io_state` WindowDevice) (Close_windows windows)
		  where
		  (windows, io_state`) = IOStateGetDevice io_state WindowDevice


Close_windows:: !(DeviceSystemState s) -> WindowHandles s
Close_windows (WindowSystemState windows)
		= Close_windows` windows

Close_windows`:: !(WindowHandles s) -> WindowHandles s
Close_windows` [(w_def, window) : windows]
		= Evaluate_2 (Close_windows` windows) (Close_window window)
Close_windows` windows
		= windows

Close_window:: !Window -> Window
Close_window window
		= Evaluate_2 window (Evaluate_2 winptr1 dispose2)
		  where
		  winptr   = WGetWindowPtr window
		  winptr1  = OSWinDispose winptr
		  dispose2 = OSCursorDispose (WGetCursorPtr window)


WindowDeviceNotEmpty:: !(DeviceSystemState s) -> Bool
WindowDeviceNotEmpty (WindowSystemState [])
		= False
WindowDeviceNotEmpty device
		= True


/* Handling all window I/O.
		  First we check whether it is a window event and next what window event.
*/
WindowIO :: !.Event !*s !(IOState *s) -> (!Bool, !*s, !IOState *s)
WindowIO e =: (OSWindowDevice,e1,e2,e3) s io
		= (True, s`, io`)
		  where
		  (s`, io`) = HandleWindowEvent e s io
WindowIO e s io
		= (False, s, io)


HandleWindowEvent:: !Event !*s !(IOState *s) -> (!*s, !IOState *s)
HandleWindowEvent (e0, OSWindowUpdate, w, e3) s io_state
		= EndWindowUpdate (s`, io_state`) newpicture w
		  where
		  (s`, dfs)           = do_update area s
		  (picture,area)      = StartWindowUpdate w pic windows
		  newpicture          = Draw_in_picture picture dfs
		  (update,pic)        = GetWindowUpdateFunction w windows
		  (windows,io_state`) = IOStateGetDevice io_state WindowDevice
		  
		  do_update [] s   = (s, [])
		  do_update area s = update area s
HandleWindowEvent event =: (e0, OSWindowActivate, w, e3) s io_state
		= let!
			io_state`` = SetActiveWindowHandle w io_state`
		  in WindowAttIO event attributes offset s (Evaluate_2 io_state`` (OSCursorSetCurrent cursor))
		  where
		  (attributes,cursor,offset) = GetWindowAttributes w windows
		  (windows,io_state`)        = IOStateGetDevice io_state WindowDevice
HandleWindowEvent event =: (e0, OSWindowMouseMove, w, e3) s io_state
		= let!
			cursor2 = cursor
		  in (s, Evaluate_2 io_state` (OSCursorSet cursor2))
		  where
		  (_,cursor,_)        = GetWindowAttributes w windows
		  (windows,io_state`) = IOStateGetDevice io_state WindowDevice
HandleWindowEvent (e0, OSWindowScroll, w, scrolltype) s io_state
		= let!
			windows`            = ScrollTheWindow scrolltype w windows
		  in (s, IOStateSetDevice io_state` windows`) 
		  where
		  (windows,io_state`) = IOStateGetDevice io_state WindowDevice
HandleWindowEvent (e0, OSWindowResize, w, flags) s io_state
		= (s, IOStateSetDevice io_state` windows`) 
		  where
		  windows`             = ResizeTheWindow flags w windows
		  (windows, io_state`) = IOStateGetDevice io_state WindowDevice
HandleWindowEvent (e0, OSWindowMaximize, w, flags) s io_state 
		= (s, IOStateSetDevice io_state` windows`)
		  where
		  windows`            = MaximizeWindow flags w windows
		  (windows,io_state`) = IOStateGetDevice io_state WindowDevice
HandleWindowEvent (e0, OSWindowRestore, w, flags) s io_state
		= (s, IOStateSetDevice io_state` windows`)
		  where
		  windows`            = RestoreWindow flags w windows
		  (windows,io_state`) = IOStateGetDevice io_state WindowDevice
HandleWindowEvent event =: (e0, e1, w, e3) s io_state
		= WindowAttIO event attributes offset s io_state`
		  where
		  (attributes,_,offset) = GetWindowAttributes w windows
		  (windows,io_state`)   = IOStateGetDevice io_state WindowDevice


WindowIsStandby:: ![WindowAttribute *s (IOState *s)] -> Bool
WindowIsStandby [StandByWindow : atts] = True
WindowIsStandby []                     = False
WindowIsStandby [x : atts]             = WindowIsStandby atts


WindowAttIO:: !Event ![WindowAttribute *s (IOState *s)] !Offset !*s !(IOState *s) -> (!*s, !IOState *s)
WindowAttIO (e0, e1, w, e3) atts offset s io
	| (e1 == OSWindowMouseActivate) && (WindowIsStandby atts)   
		= WindowAttIO` (e0, OSWindowMouse, w, e3) atts offset s io
		= WindowAttIO` (e0,e1,w,e3) atts offset s io

WindowAttIO`:: !Event ![WindowAttribute *s (IOState *s)] !Offset !*s !(IOState *s) -> (!*s, !IOState *s)
WindowAttIO` (e0, OSWindowKeyboard, w, e3) [Keyboard a f : atts] offset s io
	| SelectStateEqual a Able
		= let!  strict1  = OSGetKeyboardState w
		  in f key_info s io
		  with key_info = EventToKeyboard strict1
		= (s, io)  
WindowAttIO` (e0, OSWindowMouse, w, e3) [Mouse a f : atts] offset s io
	| SelectStateEqual a Able
		= let! strict1 = OSGetMouseState w
		  in f mouse_info s io
		  with mouse_info = EventToMouse strict1 offset
		= (s, io)        
WindowAttIO` (e0, OSWindowClose,     w, e3) [GoAway    f : atts] offset s io
		= f s io
WindowAttIO` (e0, OSWindowActivate,  w, e3) [Activate    f : atts] offset s io
		= f s io
WindowAttIO` (e0, OSWindowDeactivate, w, e3) [Deactivate f : atts] offset s io
		= f s io
WindowAttIO` event [att : atts] offset s io
		= WindowAttIO` event atts offset s io
WindowAttIO` event atts offset s io
		= (s, io)


EventToKeyboard :: !KeyEvent -> KeyboardState
EventToKeyboard (key, OSKeyUp,        mods)   = (toChar key, KeyUp, ToBools mods)
EventToKeyboard (key, OSKeyDown,        mods) = (toChar key, KeyDown, ToBools mods)
EventToKeyboard (key, OSKeyStillDown, mods)   = (toChar key, KeyStillDown, ToBools mods)

EventToMouse:: !MouseEvent !Offset -> MouseState
EventToMouse (x,y,OSMouseUp,          mods) (x_o, y_o)
		= ((x_o + x, y_o + y),ButtonUp, ToBools mods)
EventToMouse (x,y,OSMouseDown,      mods) (x_o, y_o)
		= ((x_o + x, y_o + y),ButtonDown, ToBools mods)
EventToMouse (x,y,OSMouseStillDown, mods) (x_o, y_o)
		= ((x_o + x, y_o + y),ButtonStillDown, ToBools mods) 
EventToMouse (x,y,OSDoubleClick,     mods) (x_o, y_o)
		= ((x_o + x, y_o + y),ButtonDoubleDown, ToBools mods)


ToBools:: !(!Int, !Int, !Int) -> (!Bool, !Bool, !Bool, !Bool)
ToBools (m1, m2, m3)
		= (I2B m1, I2B m2, I2B m3, I2B m3)
		  where
		  I2B 0 = False
		  I2B x = True


ScrollWindowIfNeeded :: !WindowPtr !Int !Int -> WindowPtr
ScrollWindowIfNeeded wptr 0 0
		= wptr
ScrollWindowIfNeeded wptr hdiff vdiff
		= OSWinScroll wptr hdiff vdiff


ScrollTheWindow:: !Int !WindowPtr !(DeviceSystemState s) -> DeviceSystemState s
ScrollTheWindow scrollevent w (WindowSystemState windows)
		= let!
			window = ScrollTheWindow` scrollevent w windows
		  in WindowSystemState window

ScrollTheWindow`:: !Int !WindowPtr !(WindowHandles s) -> WindowHandles s
ScrollTheWindow` scrollevent w windows
		= PutWindow w window` windows
		  where
		  window` = ScrollTheWindow`` scrollevent window
		  window  = GetWindow w windows

ScrollTheWindow``:: !Int !Window -> Window
ScrollTheWindow`` scrollevent (windowptr,pic,hscroll,vscroll,cursor,offset)
		= let!
			windowptr`                = ScrollWindowIfNeeded windowptr hdiff` vdiff`
			(hdiff`, hpos`, hscroll`) = SetScrollPosition hdiff hpos hscroll
			(vdiff`, vpos`, vscroll`) = SetScrollPosition vdiff vpos vscroll
		  in (windowptr`,pic,hscroll`,vscroll`,cursor,(hpos`,vpos`))
		  where
		  (hdiff,vdiff)     = GetScrollingValues scrollevent hscroll vscroll offset
		  (hpos, vpos)      = offset


GetScrollingValues:: !Int !ScrollState !ScrollState !Offset -> (!Int,!Int)
GetScrollingValues OSScrollLineUp  hscroll vscroll  (hpos,vpos)
		= (0, max (vscroll.scroll_min - vpos) (0 - vscroll.scroll_line))
GetScrollingValues OSScrollLineDown  hscroll vscroll (hpos,vpos)
		= (0, min (vscroll.scroll_max - vpos) vscroll.scroll_line)
GetScrollingValues OSScrollLineLeft  hscroll vscroll (hpos,vpos)
		= (max (hscroll.scroll_min - hpos) (0 - hscroll.scroll_line), 0)
GetScrollingValues OSScrollLineRight hscroll vscroll (hpos,vpos)
		= (min (hscroll.scroll_max - hpos) hscroll.scroll_line, 0)
GetScrollingValues OSScrollPageUp    hscroll vscroll (hpos,vpos)
		= (0, max (vscroll.scroll_min - vpos) (0 - vscroll.scroll_page))
GetScrollingValues OSScrollPageDown  hscroll vscroll (hpos,vpos)
		= (0, min (vscroll.scroll_max - vpos) vscroll.scroll_page)
GetScrollingValues OSScrollPageLeft  hscroll vscroll (hpos,vpos)
		= (max (hscroll.scroll_min - hpos) (0 - hscroll.scroll_page), 0)
GetScrollingValues OSScrollPageRight hscroll vscroll (hpos,vpos)
		= (min (hscroll.scroll_max - hpos) hscroll.scroll_page, 0)
GetScrollingValues OSScrollVSliderPos hscroll vscroll (hpos,vpos)
		= let!
			vpos` = (OSWinGetScrollPosition vscroll.scroll_ptr) << vscroll.scroll_div
		  in (0,  vpos` + vscroll.scroll_min - vpos)
GetScrollingValues OSScrollHSliderPos hscroll vscroll (hpos,vpos)
		= let!
			hpos` = (OSWinGetScrollPosition hscroll.scroll_ptr) << vscroll.scroll_div
		  in (hpos` + hscroll.scroll_min - hpos, 0)


SetScrollPosition:: !Int !Int !ScrollState -> (!Int, !Int, !ScrollState)
SetScrollPosition 0 pos scroll
		= (0, pos,scroll)
SetScrollPosition diff pos scroll =: {scroll_ptr,scroll_line, scroll_min, scroll_max, scroll_div}
	| diff` == 0
		= (0, pos, scroll)
		= let!
			scrollptr` = OSWinSetScrollPosition scroll_ptr spos`
		  in (diff`, pos``, {scroll & scroll_ptr = scrollptr`})
		  where
		  pos`       = pos + diff
		  pos``      = Align_thumb pos` scroll_min scroll_max scroll_line
		  diff`      = pos`` - pos
		  spos       = (pos`` - scroll_min) >> scroll_div
		  spos`      = if (scroll_div > 0 && pos`` >= scroll_max) (spos + 1) spos



RestoreWindow:: !Int !WindowPtr !(DeviceSystemState s) -> DeviceSystemState s
RestoreWindow flags w (WindowSystemState windows)
		= let!
			handles = RestoreWindow` flags w windows
		  in WindowSystemState handles

RestoreWindow`:: !Int !WindowPtr (WindowHandles s) -> WindowHandles s
RestoreWindow` flags w [wdef =: (windef,window) : windows]
	| w == WGetWindowPtr window
		= let!
			handle = RestoreWindow`` flags wdef
		  in [handle : windows]
		= let!
			handles = RestoreWindow` flags w windows
		  in [wdef : handles]
RestoreWindow` flags w windows
		= windows

RestoreWindow``:: !Int !(WindowHandle s) -> WindowHandle s
RestoreWindow`` flags (wdef =: ScrollWindow id pos t hbar vbar picdomain minsize is upd att, window)
		= let!
			wptr`             = OSWindowRes wptr
			(hwin, vwin,_,_)  = OSWinGetSizeAndPos wptr`
			(hscroll`, xpos`) = SetScrollValues hscrollptr x x2 xpos hwin hline
			(vscroll`, ypos`) = SetScrollValues vscrollptr y y2 ypos vwin vline
		  in (wdef, window`)
		  where
		  (wptr,pic,hscroll,vscroll,cursor,offset) = window
		  (xpos, ypos) = offset
		  window`      = (wptr`, pic, hscroll`, vscroll`, cursor, (xpos`, ypos`))
		  hscrollptr   = hscroll.scroll_ptr
		  hline        = hscroll.scroll_line
		  vscrollptr   = vscroll.scroll_ptr
		  vline        = vscroll.scroll_line
		  (xy, xy2)    = picdomain
		  (x,y)        = xy
		  (x2,y2)      = xy2
RestoreWindow`` flags (wdef =: FixedWindow id pos title domain upd atts, window)
		= let! wptr` = OSWindowRes wptr
		  in (wdef,window`)  
		  where
		  (wptr,pic,hscroll,vscroll,cursor,offset) = window
		  window` = (wptr`,pic,hscroll,vscroll,cursor,offset)


MaximizeWindow:: !Int !WindowPtr !(DeviceSystemState s) -> DeviceSystemState s
MaximizeWindow flags w (WindowSystemState windows) 
		= let!
			strict1 = MaximizeWindow` flags w windows
		  in WindowSystemState (strict1)

MaximizeWindow`:: !Int !WindowPtr (WindowHandles s) -> WindowHandles s
MaximizeWindow` flags w [wdef =: (windef,window) : windows]
	| w == WGetWindowPtr window
		= let!
			strict1 = MaximizeWindow`` flags wdef
		  in [strict1 : windows]
		= let!
			strict2 = MaximizeWindow` flags w windows
		  in [wdef : strict2]
MaximizeWindow` flags w windows 
		= windows

MaximizeWindow``:: !Int !(WindowHandle s) -> WindowHandle s
MaximizeWindow`` flags (wdef =: ScrollWindow id pos t hbar vbar picdomain minsize is upd att, window) 
		= let!
			wptr`               = OSWindowMax wptr True (min x2` maxw) (min y2` maxh) 0 0 
			(hwin, vwin,_,_)    = OSWinGetSizeAndPos wptr`
			(hscroll`, xpos`)   = SetScrollValues hscrollptr x x2 xpos hwin hline
			(vscroll`, ypos`)   = SetScrollValues vscrollptr y y2 ypos vwin vline
			wptr``              = ScrollWindowIfNeeded wptr` (xpos` - xpos) (ypos` - ypos)
		  in (wdef,window`)
		  where
		  (wptr,pic,hscroll,vscroll,cursor,offset) = window 
		  window`             = (wptr``,pic,hscroll`,vscroll`,cursor, (xpos`, ypos`))
		  ((x,y),(x2,y2))     = picdomain
		  (xpos, ypos)        = offset
		  (maxw, maxh)        = MaxScrollWindowSize
		  hscrollptr          = hscroll.scroll_ptr
		  hline               = hscroll.scroll_line
		  vscrollptr          = vscroll.scroll_ptr
		  vline               = vscroll.scroll_line
		  x2`                 = x2 - x 
		  y2`                 = y2 - y 
MaximizeWindow`` flags (wdef =: FixedWindow id pos title domain upd atts,window)
		= let!
			wptr` = OSWindowMax wptr False 0 0 0 0
		  in (wdef,window`)
		  where
		  (wptr,pic,hscroll,vscroll,cursor,offset) = window
		  window` = (wptr`,pic,hscroll,vscroll,cursor,offset)


ResizeTheWindow :: !Int !WindowPtr !(DeviceSystemState s) -> DeviceSystemState s
ResizeTheWindow flags w (WindowSystemState windows)
		= let!
			windows` = ResizeTheWindow` flags w windows
		  in WindowSystemState windows`

ResizeTheWindow`    :: !Int !WindowPtr !(WindowHandles s) -> WindowHandles s
ResizeTheWindow` flags w [wdef =: (windef, window) : windows]
	| w == WGetWindowPtr window
		= let!
			strict1 = ResizeTheWindow`` flags wdef
		  in [strict1 : windows]
		= let!
			strict2 = ResizeTheWindow` flags w windows
		  in [wdef : strict2]
ResizeTheWindow` flags w windows
		= windows

ResizeTheWindow``   :: !Int !(WindowHandle s) -> WindowHandle s
ResizeTheWindow`` flags (wdef =: ScrollWindow id pos t hbar vbar picdomain minsize is upd att, window)
		= let!
			wptr`            = OSWinResize wptr flags minx miny maxx maxy
			(hwin, vwin,_,_) = OSWinGetSizeAndPos wptr`
			(hscroll`,xpos`) = SetScrollValues hscrollptr x x2 xpos hwin hline
			(vscroll`,ypos`) = SetScrollValues vscrollptr y y2 ypos vwin vline
			wptr``           = ScrollWindowIfNeeded wptr` (xpos` - xpos) (ypos` - ypos)
		  in (wdef, window`)
		  where
		  (wptr,pic,hscroll,vscroll,cursor,offset) = window
		  window`      = (wptr``,pic,hscroll`,vscroll`,cursor,(xpos`, ypos`))
		  (minx, miny) = minsize
		  maxx         = x2 - x
		  maxy         = y2 - y
		  ((x,y),(x2,y2)) = picdomain
		  (xpos, ypos) = offset
		  hscrollptr   = hscroll.scroll_ptr
		  hline        = hscroll.scroll_line
		  vscrollptr   = vscroll.scroll_ptr
		  vline        = vscroll.scroll_line



/* Hiding and showing the window device i.e. hiding and showing all windows.
*/

HideWindow  :: !(IOState s) -> IOState s
HideWindow io_state 
		= IOStateSetDevice io_state` windows`
		  where
		  windows`             = HideWindow` windows
		  (windows, io_state`) = IOStateGetDevice io_state WindowDevice

HideWindow`:: !(DeviceSystemState s) -> DeviceSystemState s
HideWindow` (WindowSystemState windows)
		= let! strict1 = HideWindow`` windows
		  in WindowSystemState strict1

HideWindow``:: !(WindowHandles s) -> WindowHandles s
HideWindow`` [w =: (windef, window) : windows]
		= let!
			strict1 = Evaluate_2 w (OSWinHide (WGetWindowPtr window))
			strict2 = HideWindow`` windows
		  in [strict1 : strict2]
HideWindow`` windows
		= windows


ShowWindow:: !(IOState s) -> IOState s
ShowWindow io_state
		= IOStateSetDevice io_state` windows`
		  where
		  windows`             = ShowWindow` windows
		  (windows, io_state`) = IOStateGetDevice io_state WindowDevice

ShowWindow`:: !(DeviceSystemState s) -> DeviceSystemState s
ShowWindow` (WindowSystemState windows)
		= let! strict1 = (ShowWindow`` windows)
		  in WindowSystemState strict1

ShowWindow``:: !(WindowHandles s) -> WindowHandles s
ShowWindow`` [w =: (windef, window) : windows]
		= let!
			strict1 = Evaluate_2 w (OSWinShow (WGetWindowPtr window))
			strict2 = ShowWindow`` windows
		  in [strict1 : strict2]
ShowWindow`` windows
		= windows



/* Internal window handling functions.
*/

GetWindowCursor:: ![WindowAttribute s io ] -> CursorPtr
GetWindowCursor [Cursor shape : atts] = GetCursor shape
GetWindowCursor [att : atts]          = GetWindowCursor atts
GetWindowCursor atts                  = GetCursor StandardCursor


GetCursor:: !CursorShape -> CursorPtr
GetCursor StandardCursor = OSCursorGet OSCursorStandard
GetCursor BusyCursor     = OSCursorGet OSCursorBusy
GetCursor IBeamCursor    = OSCursorGet OSCursorIBeam
GetCursor CrossCursor    = OSCursorGet OSCursorCross
GetCursor FatCrossCursor = OSCursorGet OSCursorFatCross
GetCursor ArrowCursor    = OSCursorGet OSCursorArrow
GetCursor HiddenCursor   = OSCursorGet OSCursorHidden
												
SetNewCursor:: !Window !CursorShape -> Window
SetNewCursor (w,p,h,v,cursor,o) shape 
		= let!
			old = OSCursorDispose cursor
		  in (w, p, h, v, Evaluate_2 cursor` old, o)
		  where
		  cursor` = GetCursor shape

		
GetWindowDef:: !WindowPtr ![WindowHandle s] -> WindowDef s (IOState s)
GetWindowDef w [(windef, window) : windows]
	| w == WGetWindowPtr window
		= windef
		= GetWindowDef w windows

GetWindow:: !WindowPtr ![WindowHandle s] -> Window
GetWindow w [(windef, window) : windows]
	| w == WGetWindowPtr window
		= window
		= GetWindow w windows

PutWindow:: !WindowPtr !Window ![WindowHandle s] -> [WindowHandle s]
PutWindow w window [wdef =: (windef, window`) : windows]
	| w == WGetWindowPtr window` 
		= [(windef, window) : windows]
		= [wdef : PutWindow w window windows]

GetWindowAttributes:: !WindowPtr !(DeviceSystemState s) -> ([WindowAttribute s (IOState s)], CursorPtr, Offset)
GetWindowAttributes w (WindowSystemState windows) 
		= GetWindowAttributes` w windows

GetWindowAttributes`:: !WindowPtr !(WindowHandles s) -> ([WindowAttribute s (IOState s)], CursorPtr, Offset)
GetWindowAttributes` w [(windef, window) : windows]
	| w == WGetWindowPtr window
		= (attributes, cursor, offset)
		= GetWindowAttributes` w windows
		  where
		  attributes = WindowDef_Attributes windef
		  cursor     = WGetCursorPtr window
		  offset     = WGetOffset window
GetWindowAttributes` w windows 
		= ([], 0, (0,0))


GetWindowUpdateFunction:: !WindowPtr !(DeviceSystemState s) -> (UpdateFunction s, PicturePtr)
GetWindowUpdateFunction w (WindowSystemState windows)
		= GetWindowUpdateFunction` w windows
		  where
		  
		  GetWindowUpdateFunction`:: !WindowPtr !(WindowHandles s) -> (UpdateFunction s, PicturePtr)
		  GetWindowUpdateFunction` w [(windef, window) : windows]
			| w == WGetWindowPtr window
				= (WindowDef_Update windef, WGetPicturePtr window)
				= GetWindowUpdateFunction` w windows
		  GetWindowUpdateFunction` w windows 
				= abort "no update function"

NoUpdateFunc:: UpdateArea * s -> (*s, [DrawFunction])
NoUpdateFunc area s = (s, [])


GetPictureOffset:: !WindowPtr !(DeviceSystemState s) -> Offset
GetPictureOffset w (WindowSystemState windows)
		= GetPictureOffset` w windows
		  where
		  
		  GetPictureOffset`:: !WindowPtr !(WindowHandles s) -> Offset
		  GetPictureOffset` w [(windef, window) : windows]
			| w == WGetWindowPtr window
				= WGetOffset window
				= GetPictureOffset` w windows


SetActiveWindowHandle:: !WindowPtr !(IOState s) -> IOState s
SetActiveWindowHandle w io
		= IOStateSetDevice io` (SetActiveWindowHandle` w wh)
		  where
		  (wh, io`) = IOStateGetDevice io WindowDevice

SetActiveWindowHandle`:: !WindowPtr !(DeviceSystemState s) -> DeviceSystemState s
SetActiveWindowHandle` w s =: (WindowSystemState windows)
		= WindowSystemState [GetWindowHandle w windows : RemoveWindowHandle w windows]


GetWindowHandle:: !WindowPtr !(WindowHandles s) -> WindowHandle s
GetWindowHandle w [win =: (windef, window) : windows]
	| w == WGetWindowPtr window 
		= win
		= GetWindowHandle w windows
GetWindowHandle w windows 
		= abort "unknown window"


RemoveWindowHandle:: !WindowPtr !(WindowHandles s) -> WindowHandles s
RemoveWindowHandle w [win =: (windef, window) : windows]
	| w == WGetWindowPtr window
		= windows
		= let! strict1 = RemoveWindowHandle w windows
		  in [win : strict1]
RemoveWindowHandle w windows
		= windows


Align_thumb:: !Int !Int !Int !Int -> Int
Align_thumb thumb min max scroll
	| thumb == max 
		= thumb
		= min + (d_thumb - d_thumb mod scroll)
		  where
		  d_thumb = thumb - min


StartWindowUpdate:: !WindowPtr !PicturePtr !(DeviceSystemState s) -> (!Picture, !UpdateArea)
StartWindowUpdate w p windows 
		= StartUpdate w p offset
		  where
		  offset = GetPictureOffset w windows


EndWindowUpdate:: !(!*s, !IOState *s) !Picture !WindowPtr -> (!*s, !IOState *s)
EndWindowUpdate s pic w
		= Evaluate_2 s (EndUpdate pic)


Draw_in_window:: !Window ![DrawFunction] -> Window
Draw_in_window win =: (w,p,v,h,c,offset) fs
		= let!	picture = EndDrawing (Draw_in_picture (StartDrawing w p offset) fs)
		  in Evaluate_2 win picture


Draw_in_picture:: !Picture ![DrawFunction] -> Picture
Draw_in_picture pic [f : fs] = Draw_in_picture (f pic) fs
Draw_in_picture pic fs       = pic

:: ScrollVals :== (!Int,!Int)
:: Size :== (!Int,!Int)
:: Position :== (!Int,!Int)
		 

GetScrollVals:: ScrollBarDef -> ScrollVals
GetScrollVals (ScrollBar (Thumb thumb) (Scroll line))
		= (thumb, line)


ValidateWindow:: !(WindowDef s (IOState s)) -> WindowDef s (IOState s)
ValidateWindow (ScrollWindow id pos t hbar vbar pic ms is upd att)
	| IsValidDomain pic 
		= newwindow
		= abort "Error while opening a window: illegal PictureDomain"
		  where
		  newwindow                = ScrollWindow id pos` t hbar` vbar` pic ms` is`` upd att
		  (is`,_)                  = isize`
		  (pos`,_)                 = position`
		  (isize`,position`)       = DetermineWindowPosAndSize True is pos
		  hbar`                    = ScrollBar (Thumb xoff`) (Scroll hline)
		  vbar`                    = ScrollBar (Thumb yoff`) (Scroll vline)
		  (_,hline)                = scrollx
		  (_,vline)                = scrolly
		  scrollx                  = GetScrollVals hbar
		  scrolly                  = GetScrollVals vbar
		  (xoff`,yoff`)            = off`
		  (off`,_)                 = offset`
		  (is``,_)                 = isize``
		  (ms`,_)                  = msize`
		  (offset`,isize``,msize`) = DetermineWindowParams pic scrollx scrolly is` ms
ValidateWindow (FixedWindow id pos title pic upd att)
	| not (IsValidDomain pic) 
		= abort "Error while opening a window: illegal PictureDomain"
	| convert
		= ValidateWindow (ConvertToScrollWindow newwindow is`)
		= newwindow
		  where
		  newwindow             = FixedWindow id pos` title pic upd att
		  (is`,convert)         = initsize`
		  (pos`,_)              = position`
		  (initsize`,position`) = DetermineWindowPosAndSize False initsize pos
		  initsize              = (initw, inith)
		  initw                 = x2 - x
		  inith                 = y2 - y
		  (x,y)                 = xy
		  (x2,y2)               = x2y2
		  (xy,x2y2)             = pic


ConvertToScrollWindow:: !(WindowDef s (IOState s)) Size -> WindowDef s (IOState s)
ConvertToScrollWindow (FixedWindow id pos title pic upd att) initsize
		= ScrollWindow id pos title hbar vbar pic initsize initsize upd att
		  where      
		  hbar   = ScrollBar (Thumb x) (Scroll 10)
		  vbar   = ScrollBar (Thumb y) (Scroll 10)
		  (x,y)  = xy
		  (xy,_) = pic


IsValidDomain:: !PictureDomain -> Bool
IsValidDomain ((x_min, y_min), (x_max, y_max)) 
		= x_min < x_max && y_min < y_max
		
DetermineWindowPosAndSize:: !Bool !Size !Position -> (!(!Size,!Bool), !(!Position, !Bool))
DetermineWindowPosAndSize hasscrolls initsize initpos
		= (newinitsize, newpos)
		  where
		  (initw,inith)      = initsize
		  (posx,posy)        = initpos
		  newinitsize        = ((newinitw, newinith), initchanged)
		  newinitw           = min initw maxw
		  newinith           = min inith maxh
		  initchanged        = (newinitw <> initw) || (newinith <> inith) 
		  newpos             = ((newposx, newposy), poschanged)
		  newposx            = max 0 (min posx (maxw - newinitw))
		  newposy            = max 0 (min posy (maxh - newinith))
		  poschanged         = newposx <> posx || newposy <> posy 
		  maxw               = screenw - scrollxwidth
		  maxh               = screenh - (scrollywidth + (OSTitleBarWidth + OSMenuBarWidth))
		  scrollxwidth       = if hasscrolls OSXScrollBarHeight 0
		  scrollywidth       = if hasscrolls OSYScrollBarWidth 0
		  (screenw, screenh) = OSScreenSize 0

				  
DetermineWindowParams:: !PictureDomain !ScrollVals !ScrollVals !Size !Size -> (!(!Offset, !Bool), !(!Size, !Bool), !(!Size, !Bool))
DetermineWindowParams picdomain scrollx scrolly initsize minsize
		= (newoffset, newinitsize, newminsize)
		  where
		  (xy,x2y2)     = picdomain
		  (x,y)         = xy
		  (x2,y2)       = x2y2
		  (xoff,hline)  = scrollx
		  (yoff,vline)  = scrolly
		  (initw,inith) = initsize
		  (minw,minh)   = minsize
		  picw          = x2 - x
		  pich          = y2 - y
		  newinitsize   = ((newinitw, newinith), initchanged)
		  newinitw      = min picw initw
		  newinith      = min pich inith
		  initchanged   = newinitw <> initw || newinith <> inith
		  newminsize    = ((newminw, newminh), minchanged)
		  newminw       = max 10 (min initw minw)
		  newminh       = max 10 (min inith minh)
		  minchanged    = newminw <> minw || newminh <> minh 
		  newoffset     = ((newxoff`, newyoff`), offchanged)
		  newxoff`      = Align_thumb newxoff x x2 hline
		  newyoff`      = Align_thumb newyoff y y2 vline
		  newxoff       = max x (min xoff (x2 - newinitw))
		  newyoff       = max y (min yoff (y2 - newinith))
		  offchanged    = newxoff <> xoff || newyoff <> yoff 


MaxScrollWindowSize:: (!Int, !Int)
MaxScrollWindowSize
		= ((s_r - OSYScrollBarWidth),(s_b - (OSXScrollBarHeight + OSTitleBarWidth + OSMenuBarWidth)))
		  where
		  (s_r, s_b) = OSScreenSize 0


MaxFixedWindowSize:: (!Int,!Int)
MaxFixedWindowSize = (s_r,(s_b - (OSTitleBarWidth + OSMenuBarWidth)))
		  where
		  (s_r,s_b) = OSScreenSize 0


WindowDef_WindowId:: !(WindowDef s io) -> WindowId
WindowDef_WindowId (ScrollWindow id pos title sh sv pic msize isize upd att)
		= id
WindowDef_WindowId (FixedWindow id pos title pic upd att) 
		= id

WindowDef_Position:: !(WindowDef s io) -> WindowPos
WindowDef_Position (ScrollWindow id pos title sh sv pic msize isize upd att) 
		= pos
WindowDef_Position (FixedWindow id pos title pic upd att) 
		= pos

WindowDef_Domain:: !(WindowDef s io) -> PictureDomain
WindowDef_Domain (ScrollWindow id pos title sh sv  pic msize isize upd att)
		= pic
WindowDef_Domain (FixedWindow id pos title pic upd att) 
		= pic

WindowDef_Title:: !(WindowDef s io) -> String
WindowDef_Title (ScrollWindow id pos title sh sv pic msize isize upd att) 
		= title 
WindowDef_Title (FixedWindow id pos title pic upd att)  
		= title

WindowDef_ScrollBars:: !(WindowDef s io) -> (!ScrollBarDef, !ScrollBarDef)
WindowDef_ScrollBars (ScrollWindow id pos title sh sv pic msize isize upd att)
		= (sh,sv)

WindowDef_MinimumWindowSize:: !(WindowDef s io) -> MinimumWindowSize
WindowDef_MinimumWindowSize (ScrollWindow id pos title sh sv pic msize isize upd att) 
		= msize

WindowDef_InitialSize:: !(WindowDef s io) -> InitialWindowSize
WindowDef_InitialSize (ScrollWindow id pos title sh sv pic msize isize upd att) 
		= isize

WindowDef_Update:: !(WindowDef s io) -> UpdateFunction s
WindowDef_Update (ScrollWindow id pos title sh sv pic msize isize upd att) 
		= upd
WindowDef_Update (FixedWindow id pos title pic upd att) 
		= upd

WindowDef_Attributes:: !(WindowDef s io) -> [!WindowAttribute s io]
WindowDef_Attributes (ScrollWindow id pos title sh sv pic msize isize upd att) 
		= att
WindowDef_Attributes (FixedWindow id pos title pic upd att) 
		= att

WindowDef_SetTitle:: !(WindowDef s io) !WindowTitle -> WindowDef s io
WindowDef_SetTitle (ScrollWindow id pos title sh sv pic msize isize upd att) new_title
		= ScrollWindow id pos new_title sh sv pic msize isize upd att
WindowDef_SetTitle (FixedWindow id pos title pic upd att) new_title
		= FixedWindow id pos new_title pic upd att

WindowDef_SetUpdate:: !(WindowDef s io) !(UpdateFunction s) -> WindowDef s io
WindowDef_SetUpdate (ScrollWindow id pos title sh sv pic msize isize upd att) f_new
		= ScrollWindow id pos title sh sv pic msize isize f_new att
WindowDef_SetUpdate (FixedWindow id pos title pic upd att) f_new
		= FixedWindow id pos title pic f_new att

WindowDef_SetPictureDomain:: !(WindowDef s io) !PictureDomain -> WindowDef s io
WindowDef_SetPictureDomain (ScrollWindow id pos title sh sv pic msize isize upd att) newpic
		= ScrollWindow id pos title sh sv newpic msize isize upd att
WindowDef_SetPictureDomain (FixedWindow id pos title pic upd att) newpic
		= FixedWindow id pos title newpic upd att

WindowDef_SetAttributes:: !(WindowDef s io) ![WindowAttribute s io] -> WindowDef s io
WindowDef_SetAttributes (ScrollWindow id pos title sh sv pic msize isize upd att) newatt
		= ScrollWindow id pos title sh sv pic msize isize upd newatt
WindowDef_SetAttributes (FixedWindow id pos title pic upd att) newatt
		= FixedWindow id pos title pic upd newatt