definition module ioTypes


//////////////////////////////////////////////////
//                                               /
// Common types for the event I/O system.        /
//                                               /
// / / / / / / / / / / / / / / / / / / / / / / / /


import StdEnv
import picture
import intrface


  //-------------------------------------------------//
 //  The IO System                                  //
//-------------------------------------------------//

//
//	Definition of IOState: the environment parameter on which all
//									I/O functions operate.
//	Definition of EVENTS : the environment parameter to start I/O.
//
//	The operations on the IOState to create interactions.
//
:: *EVENTS 
:: *IOState *s 


//	OpenEvents retrieves the event stream from the world.

OpenEvents :: !*World -> (!*EVENTS, !*World)


 /*	CloseEvents replaces the event stream in the world. Event streams that
 	have been retrieved from the world, must be closed before they can be
 	opened again, otherwise a run-time error will occur. */
	
CloseEvents :: !*EVENTS !*World -> *World


:: IOSystem *s *io :== [ DeviceSystem s io ]

:: DeviceSystem *s *io =  MenuSystem   [MenuDef   s io]
				 		| TimerSystem  [TimerDef  s io]
						| WindowSystem [WindowDef s io]
						| DialogSystem [DialogDef s io]



  //-------------------------------------------------//
 //  Common definitions and simple functions        //
//-------------------------------------------------//




:: ItemTitle     :== String

:: SelectState =  Able | Unable
:: MarkState   =  Mark | NoMark

:: KeyboardState :== (!KeyCode, !KeyState, !Modifiers)
:: KeyCode       :== Char
:: KeyState      =  KeyUp | KeyDown | KeyStillDown

:: MouseState    :== (!MousePosition, !ButtonState, !Modifiers)
:: MousePosition :== (!Int, !Int)
:: ButtonState   =  ButtonUp | ButtonDown | ButtonDoubleDown |
                    ButtonTripleDown | ButtonStillDown

/* Modifiers indicates the meta keys that have been pressed (TRUE)
   or not (FALSE): (Shift, Alternate/Meta, Control, Control)
*/

:: Modifiers     :== (!Bool,!Bool,!Bool,!Bool)

:: PictureDomain :== Rectangle


  //----------------------------------------------//
 //  The type definitions for the menu device.   //
//----------------------------------------------//

/*  The menu device may consist of several PullDownMenus. The PullDownMenus
	are logically grouped into a menu bar, in the same order as they are
	specified. PullDownMenus are selected by pressing the mousebutton on
	their MenuTitle in the MenuBar. Menus contain MenuElements. The
	MenuFunction of an element is executed when the element is selected.
*/
    
:: MenuDef *s *io
       =  PullDownMenu MenuId MenuTitle SelectState [MenuElement s io]

:: MenuElement *s *io
       =  MenuItem MenuItemId ItemTitle KeyShortcut SelectState (MenuFunction s io)
         |  CheckMenuItem MenuItemId ItemTitle KeyShortcut SelectState MarkState
                          (MenuFunction s io)
         |  SubMenuItem MenuId ItemTitle SelectState [MenuElement s io]
         |  MenuItemGroup MenuItemGroupId [MenuElement s io]
         |  MenuRadioItems MenuItemId [RadioElement s io]
         |  MenuSeparator

:: RadioElement *s *io
       =  MenuRadioItem MenuItemId ItemTitle KeyShortcut SelectState
                        (MenuFunction s io)


:: MenuTitle             :== String
:: MenuFunction *s *io   :== !s -> *( !io -> (s, io)) 
:: MenuId                :== Int
:: MenuItemId            :== Int
:: MenuItemGroupId       :== Int
:: KeyShortcut           =  Key KeyCode | NoKey




  //------------------------------------------------//
 //  The type definitions for the window device.   //
//------------------------------------------------//

/*	The window device may consist of several ScrollWindows or FixedWindows.
	A ScrollWindow is defined by the following arguments:
	- WindowId			 : the number by which the programmer refers to the window.
	- WindowPos			 : the position of the upper-left corner of the window.
	- WindowTitle		 : the title of the window.
	- WindowColourType : NormalColour allows only 8 colours (see deltaPicture)
	                 			and FullColour allows all RGB colours. Don't use
	                     	FullColour windows on the Macintosh Plus.
	- ScrollBarDefs	 : the horizontal and vertical scrollbars (in that order).
	- PictureDomain	 : the range of the drawing operations in the window.
	- MinimumWindowSize: the smallest dimensions of the (contents of a) window.
	- InitialWindowSize: the initial dimensions of the (contents of a) window.
	- UpdateFunction	 : the function to redraw parts (UpdateArea) of the window.
	- An attribute list that may contain the following window attributes:
		- Activate	: the way to respond to activation of the window.
		- Deactivate: the way to respond to deactivation of the window.
		- GoAway		: the way to respond to a click in the go-away area.
		- Keyboard  : the way the window responds to keyboard input.
		- Mouse     : the way the window responds to mouse input.
		- Cursor    : the shape of the mouse pointer inside the window.
		- StandByWindow: When this attribute is present the window will be a so-called
		                 stand-by window: The mouse event related to the activation
		                 of the window will be handled also.
	A FixedWindow has a fixed size, which is defined by its PictureDomain.
	Therefore it has no scroll bars and no size parameters. When the PictureDomain
	of a FixedWindow becomes greater that on of the screen's dimensions it
	becomes a ScrollWindow.
*/

:: WindowDef * s * io
   =  ScrollWindow WindowId WindowPos WindowTitle
                   ScrollBarDef ScrollBarDef
                   PictureDomain MinimumWindowSize InitialWindowSize
                   (UpdateFunction s) [WindowAttribute s io]
   |  FixedWindow  WindowId WindowPos WindowTitle
                   PictureDomain
                   (UpdateFunction s)  [WindowAttribute s io]
 
:: WindowId          :== Int
:: WindowPos         :== (!Int, !Int)
:: WindowTitle       :== String

:: ScrollBarDef      =  ScrollBar ThumbValue ScrollValue
:: ThumbValue        =  Thumb  Int
:: ScrollValue       =  Scroll Int
:: MinimumWindowSize :== (!Int, !Int)
:: InitialWindowSize :== (!Int, !Int)

:: UpdateFunction * s      :== UpdateArea -> s -> (s,[DrawFunction]) 
:: GoAwayFunction * s * io :== s ->  *(io -> (s, io))
 
:: WindowAttribute * s * io =
      Activate   (WindowFunction s io)
   |  Deactivate (WindowFunction s io)
   |  GoAway     (WindowFunction s io)
   |  Mouse      SelectState (MouseFunction    s io)
   |  Keyboard   SelectState (KeyboardFunction s io)
   |  Cursor     CursorShape
   |  StandByWindow 

:: WindowFunction   * s * io :== s ->  *(io -> (s, io) )
:: KeyboardFunction * s * io :== KeyboardState ->  s ->  *(io -> (s, io))  
:: MouseFunction    * s * io :== MouseState ->     s ->  *(io -> (s, io))  
 

:: CursorShape =  StandardCursor | BusyCursor     | IBeamCursor |
                  CrossCursor    | FatCrossCursor | ArrowCursor | HiddenCursor


  //----------------------------------------------//
 //  The type definitions for the timer device.  //
//----------------------------------------------//


/*
	The timer device:
	The Timer responds only to timer events. A timer event occurs as
	soon as a certain TimerInterval (a number in terms of 1/TicksPerSecond
	seconds (see deltaTimer.dcl)) has expired since the last time it was
	`sampled'. The timer event causes the programmer defined TimerFunction
	to be evaluated. The TimerState argument of the TimerFunction indicates
	how many times the TimerInterval has expired since the last timer
	event. The timer device can initially be Able or Unable (SelectState).
*/

:: TimerDef * s * io		=  Timer TimerId SelectState TimerInterval (TimerFunction s io)
:: TimerId      			:== Int
:: TimerInterval			:== Int
:: TimerFunction * s * io	:== TimerState ->  s -> *( io -> (s,io) ) 
:: TimerState    			:== Int



  //------------------------------------------------//
 //  The type definitions for the dialog device.   //
//------------------------------------------------//


/*	The dialog device: Modal dialogs given in the initial dialog device are
	ignored. Use the Open(Modal)Dialog function (deltaDialog.icl) to open dialogs
	during the interaction. PropertyDialogs are special modeless dialogs with
	two predefined buttons: the Set and the Reset button. A CommandDialog can
	be modal as well as modeless.
	A PropertyDialog is defined by the following attributes:
	- DialogId: a number by which the programmer can refer to the dialog.
	- DialogTitle: The title of the dialog (ignored for modal dialogs).
	- A list of attributes that may contain the following dialog attributes:
	  - DialogPos	 : The position of the dialog on the screen.
	  - DialogSize	 : The size of the dialog.
	  - DialogMargin: The horizontal and vertical margins between the borders
	                  of the dialog and the items.
	  - ItemSpace	 : The horizontal and vertical space between the items of
	                  the dialog.
	  - StandByDialog: When this attribute is present the dialog will be a so-called
	                   stand-by dialog: it will also react to the MouseDown related
	                   to activation of the dialog.
	  When none of these attributes is specified the dialog is centered on
	  the screen, a size is chosen such that all items fit in the dialog and
	  safe default margins and item spaces are chosen. The first Measure always
	  is the horizontal attribute value, the second is always the vertical
	  attribute value.
	- SetFunction/ResetFunction: The button function for the set/reset button.
	- A list of DialogItems		: Other items such as CheckBoxes, Control's etc..
	A CommandDialog also has an id, a title, a position, a size and a list of
	DialogItems. Furthermore it has the following attribute:
	- DialogItemId: The item id of the default button.
	In the AboutDialog information about the application (version, authors etc.)
	can be presented. The first AboutDialog encountered in the initial
	DialogDevice becomes the AboutDialog of the application. Attempts to open
	AboutDialogs with OpenDialog are ignored. The AboutDialog may contain a button
	which should provide a help facility. The AboutDialog will be accessible by
	the user during the interaction in a system-dependent way.
*/

:: * DialogState * s * io 
:: DialogInfo

:: ItemInfo = EditInfo !String
            | RadioGroupInfo !DialogItemId
			| CheckGroupInfo ![ (!DialogItemId, !Bool) ]
			| ControlStateInfo !ControlState 


:: DialogDef * s * io
   =  PropertyDialog DialogId DialogTitle [DialogAttribute] (SetFunction s io)
                     (ResetFunction s io) [DialogItem s io]
   |  CommandDialog  DialogId DialogTitle [DialogAttribute]
                     DialogItemId [DialogItem s io]
   |  AboutDialog    ApplicationName PictureDomain [DrawFunction]
                     (AboutHelpDef s io)

:: ApplicationName :== String
:: AboutHelpDef * s * io = NoHelp
                           | AboutHelp ItemTitle (AboutHelpFunction s io)
:: AboutHelpFunction * s * io :== s ->  *(io -> (s,io)) 

:: DialogAttribute
   =  DialogPos    Measure Measure
   |  DialogSize   Measure Measure
   |  DialogMargin Measure Measure
   |  ItemSpace    Measure Measure
   |  StandByDialog

:: DialogId    :== Int
:: DialogTitle :== String
:: Measure     =   MM Real | Inch Real | Pixel Int
 
/*	A DialogItem can be a final button (DialogButton), a final button with a
	user-defined look (DialogIconButton), an unchangable piece of text
	(StaticText), a changeable piece of text (DynamicText), an editable text
	field (EditText), a group of RadioButtons, a group of CheckBoxes, or a
	user-defined Control. The ItemPos specifies the position of the item
	relative to the other items. When the ItemPos is DefPos the item is placed
	beneath all other items, left-aligned. 
*/ 
:: DialogItem * s * io
   =  DialogButton DialogItemId ItemPos ItemTitle SelectState (ButtonFunction s io)
   |  DialogIconButton DialogItemId ItemPos PictureDomain IconLook SelectState (ButtonFunction s io)
   |  StaticText   DialogItemId ItemPos String
   |  DynamicText  DialogItemId ItemPos TextWidth String
   |  EditText     DialogItemId ItemPos TextWidth NrEditLines String
   |  DialogPopUp  DialogItemId ItemPos SelectState DialogItemId [RadioItemDef s io]
   |  RadioButtons DialogItemId ItemPos RowsOrColumns DialogItemId [RadioItemDef s io]
   |  CheckBoxes   DialogItemId ItemPos RowsOrColumns [CheckBoxDef s io]
   |  Control      DialogItemId ItemPos PictureDomain SelectState ControlState
                       ControlLook ControlFeel (DialogFunction s io)
 
:: DialogItemId  :== Int
:: RowsOrColumns =  Rows Int | Columns Int
:: ItemPos       =  Left | Center | Right        |
                    RightTo DialogItemId         |
                    Below DialogItemId           |
                    XOffset DialogItemId Measure |
                    YOffset DialogItemId Measure |
                    XY Measure Measure           |
                    ItemBox Int Int Int Int
:: IconLook      :== SelectState -> [DrawFunction]
:: TextWidth     :== Measure
:: NrEditLines   :== Int
:: RadioItemDef * s * io =  RadioItem DialogItemId ItemTitle SelectState (DialogFunction s io)
:: CheckBoxDef * s * io  =  CheckBox DialogItemId ItemTitle SelectState MarkState (DialogFunction s io)


/*  Attributes of a user-defined control: The ControlState can be a boolean, an
	integer, a real, a string or a pair or list of one of these basic types.
	The look of the Control is defined by the list of drawfunctions returned by
	the ControlLook function. The ControlFeel defines the way to respond to mouse
	clicks in the Control's picture domain. 
*/
:: ControlState =  IntCS    Int                       |
                   BoolCS   Bool                      |
                   RealCS   Real                      | 
                   StringCS String                    |
                   PairCS   ControlState ControlState |
                   ListCS   [ControlState]
:: ControlLook :== SelectState ->  ControlState -> [DrawFunction] 
:: ControlFeel :== MouseState ->   ControlState -> (ControlState,[DrawFunction]) 
 
:: SetFunction    * s * io :== ButtonFunction s io
:: ResetFunction  * s * io :== ButtonFunction s io
:: DialogFunction * s * io :== DialogInfo -> (DialogState s io) -> DialogState s io
:: ButtonFunction *s *io :== DialogInfo -> ( s -> * (io -> (s,io)))

/*  A notice is a simple, modal dialog containing only text and final buttons.
	It can be used to inform the user about unusual or dangerous situations.
	Notices can be opened with the OpenNotice function (deltaDialog) A notice
	is defined by the following attributes:
	- A list of strings: Each string is a line of the message of the notice.
	- A NoticeButtonDef: The default button of the notice.
	- A list of NoticeButtonDefs: The other buttons of the notice. 
*/ 
:: NoticeDef       =  Notice [String] NoticeButtonDef [NoticeButtonDef]
:: NoticeButtonDef =  NoticeButton NoticeButtonId ItemTitle
:: NoticeButtonId  :== Int












//////-----------------------------------------------------------------------//
/////                                                                       ///
////   Definitions for the internal administration                         ////
///                                                                       /////
//-----------------------------------------------------------------------//////



:: IOadmin *s = { io_quit            :: !Bool,
			 	  io_menuState       :: !Perhaps (MenuWinAdmin s),
				  io_windowState     :: ![(WindowAdmin s)], 
				  io_timers          :: ![TimerAdmin s],
				  io_dialogs         :: ![DialogAdmin s],
				  io_noticeresult    :: !Perhaps NoticeButtonId,
				  io_ddehandler      :: !String -> (s -> *((IOState s) -> (s,IOState s))),
/* RWS ... */
				  io_clipboardChange :: (s -> *((IOState s) -> (s,IOState s))),
				  io_hasWorld        :: Bool,
/* ... RWS */
				  io_delayedmessages :: ![ CrossCallInfo ]
			    }



PackIOState   :: !(IOadmin s) !OS -> IOState s

UnpackIOState :: !(IOState s) -> (!IOadmin s, !OS)

UnpackIOStateWithCheck :: !(IOState s) -> (!IOadmin s, !OS)

EventsToOs :: !EVENTS -> OS

OsToEvents :: !OS -> EVENTS

ChangeOs :: !(OS -> OS) !(IOState s) -> (IOState s) 

EmptyIOadmin :: IOadmin s

//-----------------------------------------------------------------------

:: MenuKind  =  MenuBarK
			  | PullDownK
              | SubMenuK
              | ItemGroupK
              | RadioGroupK

:: MenuItemKind  =  RadioItemK
                  | CheckItemK
                  | ItemK

:: MenuAdmin *s = {  mkind     :: !MenuKind,
				     mid       :: !Int,
				     mtitle    :: !String,
				     melements :: ![MAdmin s],
				     menabled  :: !Bool,
				     mhandle   :: !HMENU,
					 mparent   :: !HMENU,
					 mpos      :: !Int
			       }

:: ItemAdmin *s = {  ikind     :: !MenuItemKind, 
				     iid       :: !Int,
					 ihandle   :: !HITEM,
				     ititle    :: !String,
				     ikey      :: !KeyShortcut,
				     ienabled  :: !Bool,
				     imarked   :: !Bool,
				     ifunction :: !MenuFunction s (IOState s),
				 	 iparent   :: !HMENU,
					 ipos      :: !Int
			       }


:: MenuWinAdmin *s = {  mbwinhandle :: !HWND,
				 	    mbmenu      :: !MAdmin s
				      }

:: MAdmin *s =  Menu !(MenuAdmin s)
              | Item !(ItemAdmin s)  
			  | Separator

//------------------------------------------------------------------


:: WindowKind =  ScrollK
			   | FixedK


:: WindowAdmin *s = {  wkind       :: !WindowKind,
					   wid         :: !Int,
					   whandle     :: !HWND,
			           wtitle      :: !String,
					   wminwidth   :: !Int,
					   wminheight  :: !Int,
					   wupdatef    :: !UpdateFunction s,
				 	   wdomain     :: !PictureDomain,
                       wpicstate   :: !PictureState,
					   whscroll    :: !Int,
					   wvscroll    :: !Int,  
					   whthumb     :: !Int,
					   wvthumb     :: !Int,  
					   wgoaway     :: !WindowFunction s (IOState s),
					   wactivate   :: !WindowFunction s (IOState s),
					   wdeactivate :: !WindowFunction s (IOState s),
					   wkeyboardf  :: !KeyboardFunction s (IOState s),
					   wkeybable   :: !Bool,
					   wmousef     :: !MouseFunction s (IOState s),
					   wmouseable  :: !Bool,
					   wcursor     :: !CursorShape
                    }				
				      
//-----------------------------------------------------------

:: TimerAdmin *s = { tid       :: !Int,
                     thandle   :: !HITEM,
                     tinterval :: !Int,
					 tlasttime :: !Int,
                     tfunction :: !TimerFunction s (IOState s),
				     table     :: !Bool
			  	   }

//---------------------------------------------------------------------



:: DialogAdmin *s = { dId     :: !Int,
                      dHandle :: !HWND,
					  dKind   :: !DialogKind,
					  dItems  :: ![ControlAdmin s]
					}

:: DialogKind = ModalK | ModelessK

:: ControlAdmin *s = { cId      :: !Int,
                       cHandle  :: !HWND,
					   cKind    :: !ControlKind s,
					   cEnabled :: !Bool,
					   cItems   :: ![ CheckRadioAdmin s ]
				     }

:: ControlKind *s = ButtonK       !(ButtonFunction s (IOState s))
                  | IconButtonK   !PictureState !PictureDomain !IconLook !(ButtonFunction s (IOState s))
			      | StaticTextK
			      | DynamicTextK  
			      | EditTextK     !NrEditLines  !String
			      | PopupK        !DialogItemId	 
			      | RadioButtonsK !DialogItemId 
			      | CheckBoxesK   
			      | CustomK       !PictureState !PictureDomain !ControlState !ControlLook !ControlFeel !(DialogFunction s (IOState s))



:: CheckRadioAdmin *s = { caId      :: !DialogItemId,
                          caHandle  :: !HWND,
                          caKind    :: !CheckRadioKind,
						  caEnabled :: !Bool,
						  caFunct   :: !DialogFunction s (IOState s)
						}

:: CheckRadioKind = RadioK | CheckK !Bool


PackDialogInfo :: ![ !(!DialogItemId, !ItemInfo) ] -> DialogInfo
UnpackDialogInfo :: !DialogInfo -> [ ( !DialogItemId, !ItemInfo) ] 

PackDialogState :: !(DialogAdmin s) !OS -> DialogState s (IOState s)
UnpackDialogState :: !(DialogState s io) -> ( !DialogAdmin s, !OS)

  //------------------------------------------------//
 //      Helper functions                          //
//------------------------------------------------//

CursorShape2Code :: !CursorShape -> Int


SetBetween :: !Int !Int !Int -> Int


MakeMouseState :: !Int !Int !Int !Int -> MouseState

MakeKeyboardState :: !Int !Int !Int -> KeyboardState

toMods :: !Int -> Modifiers


SelectStateEqual :: !SelectState !SelectState -> Bool

MarkEqual  :: !MarkState !MarkState -> Bool

Enabled  :: !SelectState -> Bool

Marked  :: !MarkState -> Bool

:: Id   :== Int
:: Size :== (Int,Int)


:: Perhaps a  =  OK !a
			   | Nope

RectangleDimensions :: !Rectangle -> (Int,Int)

StateMap2 :: ( x s -> s) ![x] !s -> s

Select :: (x -> Bool) x ![x] -> (Bool, x)


// smap :: (a s -> (b,s)) [a] s  -> ([b],s)
smap f as s :== smap_ as s
where
	smap_ []       s  =  ([],s)
	smap_ [a:rest] s  =  ([b:restb],finals)
	where
		(b,s2)         = f a s 
		(restb,finals) = smap_ rest s2

Error :: !String !String !String -> .x

IOStateSetWorld :: *World !(IOState s) -> !IOState s
IOStateGetWorld :: !(IOState s) -> (!*World, !IOState s)
