implementation module deltaBitmap


//	Clean 0.8 I/O library.
//	Interface functions for drawing bitmaps.


import StdArray, StdBool, StdFile, StdInt, StdClass
import ioTypes, picture


BitmapError :: String String -> .x
BitmapError rule error
	= Error rule "Bitmap" error

::	Bitmap
	=	{	bitmapWidth		:: !Int			// The width of the bitmap
		,	bitmapHeight	:: !Int			// The height of the bitmap
		,	bitmapContents	:: !{#Char}		// The bitmap information
		}

getBitmapSize :: !Bitmap -> (!Int,!Int)
getBitmapSize { bitmapWidth,bitmapHeight }
	= (bitmapWidth,bitmapHeight)

openBitmap :: !{#Char} !*env -> (!(!Bool,!Bitmap),!*env)	| FileSystem env
openBitmap name env
	# (ok,file,env)		= fopen name FReadData env
	| not ok
	= ((ok,noBitmap),env)
	# (ok,bitmap,file)	= readBitmap file
    # (_,env)			= fclose file env
	= ((ok,bitmap),env)
where
	noBitmap			= { bitmapWidth=0,bitmapHeight=0,bitmapContents={} }
	
	// readBitmap reads a bitmap from a file. See page 176 of Programming Windows 95 (Charles Petzold)
	readBitmap :: !*File -> (!Bool,!Bitmap,!*File)
	readBitmap file
		# (_, c1,file) = freadc file
		# (ok,c2,file) = freadc file      // read first two bytes
		| not ok || c1<>'B' || c2<>'M'	  // are they "BM"? 
		= (False,noBitmap,file)
		# (_,  fileSize,file)	= freadi file // read file size
		# (_,  _,       file)	= freadi file // skip bfReserved1 & 2
		# (_,  _,       file)	= freadi file // skip bfOffBits
		# (_,  _,       file)	= freadi file // skip biSize
		# (_,  w,       file)	= freadi file // read width
		# (ok1,h,       file)	= freadi file // read height
		# (ok2,         file)	= fseek  file 0 FSeekSet
		| not ok1 || not ok2
		= (False,noBitmap,file)
		# (data,file)			= freads file fileSize
		| size data <> fileSize
		= (False,noBitmap,file)
		| otherwise 
		= (True,{bitmapWidth=w,bitmapHeight=h,bitmapContents=data},file) 

drawBitmap :: !Point !Rectangle !Bitmap !Picture -> Picture
drawBitmap pos part bitmap=:{ bitmapWidth,bitmapHeight,bitmapContents } picture
	| bitmapWidth==0 || bitmapHeight==0 || size bitmapContents==0
	= picture
	# (x1,y1,x2,y2)	= RectangleToRect part
	  (x1,y1,x2,y2)	= (SetBetween x1 0 bitmapWidth,SetBetween y1 0 bitmapHeight,SetBetween x2 0 bitmapWidth,SetBetween y2 0 bitmapHeight)
	| x1==x2 || y1==y2
	= picture
	# (hdc,tb)		= UnpackPicture picture
	# (hdc,tb)		= WinDrawBitmap pos part bitmapContents (hdc,tb)
	= PackPicture hdc tb
where
	RectangleToRect :: !Rectangle -> Rect
	RectangleToRect ((a,b),(a`,b`))
		| x_less_x` && y_less_y`= (a,b,a`,b`)
		| x_less_x`				= (a,b`,a`,b)
		| y_less_y`				= (a`,b,a,b`)
								= (a`,b`,a,b)
	where
		x_less_x` = a<=a`
		y_less_y` = b<=b`

WinDrawBitmap :: !(!Int, !Int) !(!(!Int,!Int),!(!Int,!Int)) !{#Char} !(!Int,!*Int) -> (!Int,!*Int)
WinDrawBitmap _ _ _ _
  = code
	{
	.inline WinDrawBitmap
		ccall WinDrawBitmap "IIIIIISII-II"
	.end
	}
