implementation module StdReal

// ********************************************************
//	Concurrent Clean Standard Library Module Version 1.3
//	Copyright 1998 University of Nijmegen
// ********************************************************

import StdOverloaded,StdInt,StdArray
from StdBool import &&,||,not
from StdClass import <=
from StdChar import ==

instance + Real
where
 (+) :: !Real !Real -> Real
 (+) a b
	= code {
		.inline +;r
			addR
		.end
	}

instance - Real
where
 (-) :: !Real !Real -> Real
 (-) a b
	= code {
		.inline -;r
			subR
		.end
	}

instance zero Real
where
 zero:: Real
 zero
	= code {
		.inline zero;r
			pushR 0.0
		.end
	}

instance * Real
where
 (*) :: !Real !Real -> Real
 (*) a b
	= code {
		.inline *;r
			mulR
		.end
	}

instance / Real
where
 (/) :: !Real !Real -> Real
 (/) a b
	= code {
		.inline /;r
			divR
		.end
	}

instance one Real
where
 one:: Real
 one
	= code {
		.inline one;r
			pushR 1.0
		.end
	}

instance ^ Real
where
 (^) :: !Real !Real -> Real
 (^) a b
	= code {
		.inline ^;r
			powR
		.end
	}

instance abs Real
where
 abs::!Real -> Real
 abs x = if (x<0.0) (0.0 - x) x

instance sign Real
where
 sign::!Real -> Int
 sign x | x == 0.0	= 0
		| x < 0.0	= -1
 					= 1

instance ~ Real
where
 ~ :: !Real -> Real
 ~ x
	= code {
		.inline ~;r
			negR
		.end
	}

instance == Real
where
 (==) :: !Real !Real -> Bool
 (==) a b
	= code {
		.inline ==;r
			eqR
		.end
	}

instance < Real
where
 (<) :: !Real !Real -> Bool
 (<) a b
	= code {
		.inline <;r
			 ltR
		.end
	}

instance ln Real
where
	ln a
		= code {
			.inline ln;r
				lnR
			.end
		}

instance log10 Real
where
	log10 a
		= code {
			.inline log10;r
				log10R
			.end
		}

instance exp  Real
where exp a
		= code {
			.inline exp;r
				expR
			.end
		}

instance sqrt Real
where sqrt a
		= code {
			.inline sqrt;r
				sqrtR
			.end
		}

instance sin Real
where
	sin a
		= code {
			.inline sin;r
				sinR
			.end
		}

instance cos Real
where
	cos a
		= code {
			.inline cos;r
				cosR
			.end
		}

instance tan Real
where 
	tan a
		= code {
			.inline tan;r
				tanR
			.end
		}
		
instance asin Real
where
	asin a
		= code {
			.inline asin;r
				asinR
			.end
		}

instance acos Real
where
	acos a
		= code {
			.inline acos;r
				acosR
			.end
		}

instance atan Real
where
	atan a
		= code {
			.inline atan;r
				atanR
			.end
			.inline abs;r
			.end
			.inline sign;r
			.end
			.inline sinh;r
			.end
			.inline cosh;r
			.end
			.inline tanh;r
			.end
			.inline asinh;r
			.end
			.inline acosh;r
			.end
			.inline atanh;r
			.end
		}

instance sinh Real
where
	sinh x = (exp x - exp (~ x)) * 0.5

instance cosh Real
where
	cosh x =  (exp x + exp (~ x)) * 0.5

instance tanh Real
where
	tanh x = (expx - expmx) / (expx + expmx)
	where
		expx = exp x
		expmx = exp (~ x)

instance asinh Real
where
	asinh x = ln (x + sqrt (x*x + 1.0))

instance acosh Real
where
	acosh x = ln (x + sqrt (x*x - 1.0))  // only the positive value is taken

instance atanh Real
where
	atanh x = 0.5 * ln ((1.0 + x)/(1.0 - x))

instance toReal	Int
where
 toReal :: !Int -> Real
 toReal a
	= code {
		.inline toReal;i
			ItoR
		.end
	}

instance toReal Real
where
 toReal :: !Real -> Real
 toReal a
	= code {
		.inline toReal;r
			 no_op
		.end
	}

instance fromReal Int
where
 fromReal :: !Real -> Int
 fromReal a
	= code {
		.inline fromReal;i
			RtoI
		.end
	}

instance fromReal Real
where
 fromReal :: !Real -> Real
 fromReal a
	= code {
		.inline fromReal;r
			 no_op
		.end
	}

instance fromReal {#Char}
where
 fromReal :: !Real -> {#Char}
 fromReal a
	= code {
		.inline fromReal;#
		.d 0 2 r
			jsr RtoAC
		.o 1 0
		.end
	}

instance toReal {#Char}
where
 toReal::!{#Char} -> Real
 toReal s
	|	len == 0
			=	0.0
	|	first  == '-'
			= 	~ signedval
	|	first  == '+'
			=	signedval
	//	otherwise
			= 	val
 where
	len
		=	size s
	signedval
		= 	toReal2 s 1 0.0 False 1.0 False 0 0
	val
		=	toReal2 s 0 0.0 False 1.0 False 0 0
	first
		=	s.[0]

	toReal2 s posn val dec_new dval exp eneg eval
		| posn == len
			= 	val*dval*10.0 ^  toReal (eneg*eval)
		| digit && not dec_new && not exp
			= 	toReal2 s (posn+1) (toReal n + 10.0*val) dec_new dval exp eneg eval
		| digit && dec_new && not exp
			= 	toReal2 s (posn+1) (toReal n + 10.0*val) dec_new (dval*0.1) exp eneg eval
		| digit && exp
			= 	toReal2 s (posn+1) val dec_new dval exp eneg (n + 10*eval )
		| not dec_new && not exp && c == '.'
			= 	toReal2 s (posn+1) val True 1.0 exp eneg eval
		| not exp && (c=='e' || c=='E')
			| posn<len-2 && s.[posn+1] == '-'
				= 	toReal2 s (posn+2) val dec_new dval True (-1) 0
			| posn<len-2 && s.[posn+1] == '+'
				= 	toReal2 s (posn+2) val dec_new dval True (+1) 0
			| posn<len-1
				= 	toReal2 s (posn+1) val dec_new dval True 1 0 
			// otherwise
				= 	0.0
		// otherwise
			= 	0.0
		where
			c		=	s.[posn]
			n		=	toInt c  -  toInt '0' 
			digit	=	0<=n  &&  n<=9 

entier :: !Real -> Int
entier a
	= code {
		.inline entier
			entierR
		.end
		.inline toReal;#
		.end
	}
